import {mapValues} from 'lodash';

import {dutySwapNotifyConfig, dutySwapReminders} from '../configs/constants';

import formatDateMonth from '../utilities/formatDateMonth';
import logger from '../utilities/logger';
import * as md from '../utilities/markdown';
import getStaffProfileUrl from '../utilities/getStaffProfileUrl';

import api from '../api';
import * as notifications from './notifications';
import * as pin from './pin';

export default async function notifySwapDuty(): Promise<void> {
    try {
        const duties = await Promise.all(
            Object.values(dutySwapNotifyConfig).map(async group => {
                const persons = await api.abc.getOnDuty(group.slug);
                const firstPerson = persons[0];

                return {
                    name: `${group.name}`,
                    start: firstPerson
                        ? formatDateMonth(new Date(firstPerson.start_datetime))
                        : null,
                    finish: firstPerson
                        ? formatDateMonth(new Date(firstPerson.end_datetime))
                        : null,
                    persons: mapValues(group.persons, role => {
                        const person = persons.find(
                            p => p.schedule.slug === role.slug,
                        );

                        if (!person) {
                            return;
                        }

                        return `${md.link(role.name, role.scheduleUrl)}: ${
                            person.person.first_name.ru
                        } ${person.person.last_name.ru} ${md.link(
                            md.escape(person.person.login),
                            getStaffProfileUrl(person.person.login),
                        )}`;
                    }),
                };
            }),
        );

        const firstDuty = duties[0];

        const message = [
            md.escape(
                `Дежурные ${
                    firstDuty
                        ? `c ${firstDuty.start} по ${firstDuty.finish}`
                        : ''
                }`,
            ),
            '',
            duties
                .map(duty => {
                    return [duty.name, ...Object.values(duty.persons)].join(
                        '\n',
                    );
                })
                .join('\n\n'),
            '',
            'А также не забываем:',
            dutySwapReminders.join('\n'),
        ].join('\n');

        const {messageId, chatId} = await notifications.onSendMessageInDutyChat(
            message,
            {
                parse_mode: 'MarkdownV2',
            },
        );

        await pin.unpinChatMessage(chatId);

        await pin.pinChatMessage(chatId, messageId, {
            disableNotification: true,
        });
    } catch (e) {
        if (e instanceof Error) {
            logger.logError('Не удалось написать о смене дежурства', e);
        }
    }
}
