import {partition} from 'lodash';

import logger from '../utilities/logger';

import api from '../api';

/**
 * Отпинить все ранее припиненые сообщения ТОЛЬКО РОБОТА в чате
 */
export async function unpinChatMessage(chatId: string): Promise<void> {
    const pinnedMessages = await api.storage.getPinnedMessages();

    const [chatMessages, otherMessages] = partition(
        pinnedMessages,
        item => item.chatId === chatId,
    );

    await Promise.all(
        chatMessages.map(async item => {
            try {
                await api.telegram.unpinChatMessage(
                    item.chatId,
                    item.messageId,
                );
            } catch (e) {
                if (e instanceof Error) {
                    logger.logError(
                        `Failed unpin message "${item.messageId}" in chat "${item.chatId}"`,
                        e,
                    );
                }
            }
        }),
    );

    await api.storage.setPinnedMessages(otherMessages);
}

/**
 * Запинить сообщение в чате и запомнить id сообщения в S3
 */
export async function pinChatMessage(
    chatId: string,
    messageId: number,
    {disableNotification}: {disableNotification?: boolean} = {},
): Promise<void> {
    const pinnedMessages = await api.storage.getPinnedMessages();

    await api.telegram.pinChatMessage(chatId, messageId, {
        disable_notification: disableNotification,
    });

    await api.storage.setPinnedMessages([
        ...pinnedMessages,
        {chatId, messageId},
    ]);
}
