import {ICarouselBlockItem} from '../../types/blocks';

// Костыль для выравнивания высоты колонок, т.к. в универсальной верстке это
// не достижимо, а из-за ограничения на размер письма мы не можем делать 2 разные
// верстки для тача и десктопа (они обе присутствовали бы в письме, но в зависимости
// от типа устройства показывалась либо одна вёрстка, либо другая)
const AVERAGE_TITLE_SYMBOLS_IN_ROW = 12;
const TITLE_ROW_HEIGHT = 22;
const AVERAGE_DESCRIPTION_SYMBOLS_IN_ROW = 18;
const DESCRIPTION_ROW_HEIGHT = 20;

const replaceSymbols = (str: string): string => str.replace(/&#?\w+;/g, '.');

export const getCarouselItemTitleRows = (items: ICarouselBlockItem[]): number =>
    Math.ceil(
        items.reduce(
            (length, {title}) => Math.max(length, replaceSymbols(title).length),
            0,
        ) / AVERAGE_TITLE_SYMBOLS_IN_ROW,
    );

export const getCarouselItemDescriptionRows = (
    items: ICarouselBlockItem[],
): number =>
    Math.ceil(
        items.reduce(
            (length, {description}) =>
                Math.max(length, replaceSymbols(description).length),
            0,
        ) / AVERAGE_DESCRIPTION_SYMBOLS_IN_ROW,
    );

export function carouselItem(
    item: ICarouselBlockItem,
    options: {
        isLast: boolean;
        withInfo: boolean;
        rows: {
            title: number;
            description: number;
        };
    },
): string {
    const {rows, isLast, withInfo} = options;
    const iconCell = item.info.icon
        ? `
        <td style="width: 24px;">
            <img
                style="width: 16px; height: 16px; vertical-align: text-bottom;"
                src="${item.info.icon}"
                height="16"
                width="16"
            />
        </td>
    `
        : '';

    const infoRow = item.info.text
        ? `
        <tr>
            ${iconCell}
            <td>
                <span style="font-size: 15px; line-height: 20px;">
                    ${item.info.text}
                </span>
            </td>
        </tr>
    `
        : '';

    return `
        <mj-column
            css-class="rounded ${isLast ? 'desktop_only' : ''}"
            width="33.333%"
            padding="0 6px 20px"
            inner-border-top="none"
        >
            <mj-button
                href="${item.url}"
                target="_blank"
                rel="noopener noreferrer"
                padding="0"
                inner-padding="0"
                background-color="#fff"
            >
                <table cellspacing="0" cellpadding="0" style="color: #000;">
                    <tr>
                        <td>
                            <img
                                class="carousel_image"
                                src="${item.image}"
                                width="202px"
                                style="display: block; border-radius: 8px 8px 0 0; background-color: #e6e6e6; object-fit: cover;"
                            />
                        </td>
                    </tr>

                    <tr>
                        <td
                            class="left-border right-border fluid-height"
                            style="height: ${
                                rows.title * TITLE_ROW_HEIGHT
                            }px; padding: 12px 16px 4px; vertical-align: top;"
                        >
                            <h3 style="margin: 0; font-size: 18px; line-height: 22px; font-weight: 700;">
                                ${item.title}
                            </h3>
                        </td>
                    </tr>

                    <tr>
                        <td
                            class="left-border right-border ${
                                withInfo
                                    ? ''
                                    : 'bottom-border bottom-border-rounded'
                            } fluid-height ${
        infoRow ? '' : 'collapseBottomPadding'
    }"
                            style="height: ${
                                rows.description * DESCRIPTION_ROW_HEIGHT
                            }px; padding: 0 16px ${
        withInfo ? '12px' : '16px'
    }; vertical-align: top;"
                        >
                            <p style="margin: 0; font-size: 15px; line-height: 20px; font-weight: 400;">
                                ${item.description}
                            </p>
                        </td>
                    </tr>

                    ${
                        withInfo
                            ? `
                        <tr>
                            <td class="left-border right-border bottom-border bottom-border-rounded">
                                <table class="fluid-height" style="padding: 0 16px 16px; height: 42px;">
                                    ${infoRow}
                                </table>
                            </td>
                        </tr>
                    `
                            : ''
                    }
                </table>
            </mj-button>
        </mj-column>
    `;
}
