/** *****************************************************************************/
/* Базовые типы                                                                 */
/** *****************************************************************************/

export enum EBlockType {
    HEADER = 'HEADER',
    FOOTER = 'FOOTER',
    GREETING = 'GREETING',
    CAROUSEL = 'CAROUSEL',
    WEATHER = 'WEATHER',
    USEFUL = 'USEFUL',
    BOOKING = 'BOOKING',
    TRAIN_TICKET = 'TRAIN_TICKET',
    BEFORE_DEPARTURE = 'BEFORE_DEPARTURE',
    DISCLAIMERS = 'DISCLAIMERS',
}

export enum EActionType {
    PRIMARY = 'PRIMARY',
    SECONDARY = 'SECONDARY',
}

export interface ILink {
    text: string;
    url: string;
}

interface IImageLink {
    image: string;
    url: string;
}

interface IButton {
    text: string;
    theme: EActionType;
}

export interface IButtonLink extends IButton, ILink {}

interface IBaseBlock {
    title: string;
    subtitle?: string;
}

interface IBaseActionBlock extends IBaseBlock {
    action: IButtonLink;
}

/** *****************************************************************************/
/* Интерфейсы блоков                                                             */
/** *****************************************************************************/

// Шапка

export interface IHeaderBlock {
    type: EBlockType.HEADER;
    /**
     * Ссылка на картинку - логотип сервиса
     */
    logo: IImageLink;
    links: ILink[];
}

// Приветствие

export interface IGreetingBlock extends IBaseBlock {
    type: EBlockType.GREETING;
    image: string;
    /**
     * Приветственный текст
     */
    text: string;
    /**
     * Определяет тип отображения
     * overlaying: false - текст располагается под изображением
     * overlaying: true - текст наложен поверх изображения
     */
    overlaying: boolean;
}

// Бронирование

export interface IBookingBlock extends IBaseBlock {
    type: EBlockType.BOOKING;
    hotelName: string;
    image: string;
    /**
     * Ссылка - адрес отеля
     */
    geoPosition: ILink;
    checkin: string;
    checkout: string;
    dates: string;
    download?: IButtonLink;
}

// Билет на поезд

export interface ITrainTicketBlock extends IBaseBlock {
    type: EBlockType.TRAIN_TICKET;
    orderNumber: string;
    /**
     * Точки маршрута (Москва - Владимир)
     */
    points: string[];
    /**
     * Полезная информация о посадке (Нужен только паспорт и т.д.)
     */
    boardingHint: string;
    departure: IRoutePoint;
    arrival: IRoutePoint;
    /**
     * Длительность поездки в удобном для человека виде (примерный формат: 7 ч 45 мин, 2 д 12 ч)
     */
    duration: string;
    /**
     * Номер поезда / номер поезда + название
     */
    trainName: string;
    /**
     * Информация о вагоне / местах
     */
    trainInfo: string;
    /**
     * Список удобств
     */
    facilities: {
        icon: string;
        /**
         * Для тех кто не понял что это за иконка должна быть возможность показать хинт
         */
        hint: string;
    }[];
    download: IButtonLink;
    /**
     * Ссылка - адрес/маршрут до вокзала
     */
    route?: ILink;
}

interface IRoutePoint {
    /**
     * Название станции отправления
     */
    station: string;
    /**
     * Дата в удобном для человека формате (22 июля, суббота)
     */
    date: string;
    /**
     * Время в формате HH:MM
     */
    time: string;
    /**
     * Приписка в случае если дата предыдущей точки маршрута отличается от текущей
     * Примеры:
     *  - поезд отправляется 22 числа, а прибывает 23
     *  - первая часть маршрута попадает в одни сутки, а следующая - в другие
     */
    nextDateHint?: string;
}

// Группа блоков "Полезное"

export interface IUsefulBlock extends IBaseBlock {
    type: EBlockType.USEFUL;
    items: IUsefulBlockItem[];
}

interface IUsefulBlockItem {
    title: string;
    description: string;
    image: string;
    url: string;
}

// Группа блоков "Карусель"

export interface ICarouselBlock extends IBaseActionBlock {
    type: EBlockType.CAROUSEL;
    items: (ICarouselBlockItem | ICarouselBlockHotelItem)[];
}

interface ICarouselBlockBaseItem {
    image: string;
    title: string;
    url: string;
}

export interface ICarouselBlockHotelItem extends ICarouselBlockBaseItem {
    stars?: number;
    rating: number;
    reviews: string;
    accommodation: string;
    orderInfo: string[];
    price: string;
}

export interface ICarouselBlockItem extends ICarouselBlockBaseItem {
    description: string;
    info: {
        icon: string;
        text: string;
    };
}

// Погода

export interface IWeatherBlock extends IBaseActionBlock {
    type: EBlockType.WEATHER;
    items: IWeatherBlockItem[];
}

export interface IWeatherBlockItem {
    /**
     * 2-х буквенное название дня недели
     */
    day: string;
    /**
     * Дата (20 янв)
     */
    isWeekend: boolean;
    date: string;
    temperature: {
        day: string;
        night: string;
    };
    /**
     * Иконка и текст для отображения погодных условий
     */
    conditions: {
        icon: string;
        description: string;
    };
}

// Перед отъездом

export interface IPreTripBlock extends IBaseBlock {
    type: EBlockType.BEFORE_DEPARTURE;
    blocks: {
        title: string;
        items: {
            icon: string;
            description: string;
        }[];
    }[];
    /**
     * Пожелания счастливого пути
     */
    partingWishes: string;
}

// Подвал

export interface IFooterBlock {
    type: EBlockType.FOOTER;
    /**
     * Ссылки на соцсети
     */
    social: IImageLink[];
    /**
     * Телефон поддержки и сопровождающий текст
     * Возможно стоит сразу упороться за транслокальность и в сопровождающий текст вставлять плейсхолдер
     * для последующей замены на ссылку с номером телефона
     */
    support: {
        phoneNumber: string;
        hint: string;
    };
    /**
     * Подвальные ссылки
     */
    links: ILink[];
}

// Дисклеймеры

export interface IDisclaimersBlock {
    type: EBlockType.DISCLAIMERS;
    disclaimers: string[];
}

/** *****************************************************************************/
/* Утилитарные типы                                                             */
/** *****************************************************************************/

export type TEmailBlock =
    | IHeaderBlock
    | IGreetingBlock
    | IBookingBlock
    | ITrainTicketBlock
    | IUsefulBlock
    | ICarouselBlock
    | IWeatherBlock
    | IPreTripBlock
    | IFooterBlock
    | IDisclaimersBlock;

export const isHeaderBlock = (block: TEmailBlock): block is IHeaderBlock =>
    block.type === EBlockType.HEADER;

export const isGreetingBlock = (block: TEmailBlock): block is IGreetingBlock =>
    block.type === EBlockType.GREETING;

export const isBookingBlock = (block: TEmailBlock): block is IBookingBlock =>
    block.type === EBlockType.BOOKING;

export const isTrainTicketBlock = (
    block: TEmailBlock,
): block is ITrainTicketBlock => block.type === EBlockType.TRAIN_TICKET;

export const isUsefulBlock = (block: TEmailBlock): block is IUsefulBlock =>
    block.type === EBlockType.USEFUL;

export const isCarouselBlock = (block: TEmailBlock): block is ICarouselBlock =>
    block.type === EBlockType.CAROUSEL;

export const isWeatherBlock = (block: TEmailBlock): block is IWeatherBlock =>
    block.type === EBlockType.WEATHER;

export const isPreTripBlock = (block: TEmailBlock): block is IPreTripBlock =>
    block.type === EBlockType.BEFORE_DEPARTURE;

export const isFooterBlock = (block: TEmailBlock): block is IFooterBlock =>
    block.type === EBlockType.FOOTER;

export const isCarouselBlockItem = (
    item: ICarouselBlockItem | ICarouselBlockHotelItem,
): item is ICarouselBlockItem => 'info' in item;

export const isCarouselHotelItem = (
    item: ICarouselBlockItem | ICarouselBlockHotelItem,
): item is ICarouselBlockHotelItem => 'accommodation' in item;

export const isDisclaimersBlock = (
    block: TEmailBlock,
): block is IDisclaimersBlock => block.type === EBlockType.DISCLAIMERS;
