import {upperFirst, camelCase} from 'lodash';
import * as crypto from 'crypto';
import {EExportMode, IFriggaOptions} from '@yandex-data-ui/frigga';

const ILLUSTRATIONS_RE =
    /^Illustrations\s*\/\s*(?<name>.*)\s*\/\s*(?<size>.*)$/;

function parseComponentName(componentName: string): {
    size: string;
    name: string;
} {
    const matchResult = ILLUSTRATIONS_RE.exec(componentName);

    if (!matchResult?.groups) {
        throw new Error('Unexpected component name');
    }

    const {size, name} = matchResult.groups;

    return {size, name};
}

export const file: IFriggaOptions['file'] = 'I8EBpOVkBHKTHVrDyWuaaIhI';

export const select: IFriggaOptions['select'] = ({name}) => {
    return ILLUSTRATIONS_RE.test(name);
};

export const getFileName: IFriggaOptions['getFileName'] = (
    component,
    exportMode,
) => {
    const extension = exportMode === EExportMode.SVG ? 'svg' : 'tsx';
    const {size, name} = parseComponentName(component.name);

    const titleCasedName = upperFirst(camelCase(name));

    return `${titleCasedName}/${titleCasedName}${size}.${extension}`;
};

export const getComponentName: IFriggaOptions['getComponentName'] =
    component => {
        const {size, name} = parseComponentName(component.name);

        return upperFirst(camelCase(`${name}${size}`));
    };

export const getSvgrConfig: IFriggaOptions['getSvgrConfig'] = _component => {
    const prefix = crypto
        .createHash('md5')
        .update(_component.name)
        .digest('hex')
        .slice(-8);

    return {
        template: ({componentName, jsx}, {tpl}) => {
            return tpl`
          import {ReactElement} from 'react';
          ${'\n'}
          import {IIconProps} from 'icons/types/icon';

          ${'\n'}
          const ${componentName} = (props: IIconProps): ReactElement => ${jsx};
          ${'\n'}
          export default ${componentName};
        `;
        },
        typescript: true,
        dimensions: false,
        svgProps: {
            focusable: 'false',
        },
        svgoConfig: {
            plugins: [
                {
                    name: 'prefixIds',
                    params: {
                        prefix,
                    },
                },
            ],
        },
    };
};
