import {MINUTE} from 'helpers/constants/dates';

import {AviaSearchForm} from 'helpers/project/avia/components/AviaSearchForm';
import TestAviaSearchResultFog from 'helpers/project/avia/pages/SearchResultsPage/components/TestAviaSearchResultFog';
import {NoResultsYak} from 'helpers/project/avia/pages/SearchResultsPage/components/NoResultsYak';
import {TestAviaTouchFilters} from 'helpers/project/avia/pages/SearchResultsPage/components/TestAviaTouchFilters';

import {Page} from 'components/Page';
import {AviaSorting} from './components/Sorting';
import {TestAviaResultVariant} from './components/TestAviaResultVariant/TestAviaResultVariant';
import {AviaDesktopFilters} from './components/TestAviaFilters/DesktopFilters';
import {Component} from 'components/Component';
import {ComponentArray} from 'components/ComponentArray';
import {TestSearchHotelsCrossSaleMap} from 'components/TestSearchHotelsCrossSaleMap';

interface IAviaSearchPoint {
    id: string;
    name: string;
}

interface IAviaSearchTravellers {
    adults: number;
    children?: number;
    infants?: number;
}

export interface IAviaSearchParams {
    from: IAviaSearchPoint;
    to: IAviaSearchPoint;
    travellers?: IAviaSearchTravellers;
    klass?: 'business' | 'economy';
    startDate: string;
    endDate?: string;
    filters?: string;
}

export class AviaSearchResultsPage extends Page {
    readonly searchForm: AviaSearchForm;
    readonly sorting: AviaSorting;
    readonly variants: ComponentArray<TestAviaResultVariant>;
    readonly fog: TestAviaSearchResultFog;
    readonly noResultsYak: NoResultsYak;
    readonly touchFiltersBlock: TestAviaTouchFilters;
    readonly subscriptionAndDynamicAsButtons: Component;
    readonly crossSaleMap: TestSearchHotelsCrossSaleMap;
    readonly aviaSearchIsCompleted: Component;

    constructor(browser: WebdriverIO.Browser) {
        super(browser);

        this.searchForm = new AviaSearchForm(browser);
        this.variants = new ComponentArray(
            browser,
            'aviaSearchVariant',
            TestAviaResultVariant,
        );
        this.sorting = new AviaSorting(browser, 'avia-sorting');

        this.fog = new TestAviaSearchResultFog(browser);
        this.noResultsYak = new NoResultsYak(browser);
        this.touchFiltersBlock = new TestAviaTouchFilters(browser);
        this.subscriptionAndDynamicAsButtons = new Component(
            browser,
            'avia-search-subscriptionAndDynamicAsButtons',
        );
        this.crossSaleMap = new TestSearchHotelsCrossSaleMap(
            browser,
            'avia-search-hotelsCrossSaleMap',
        );

        this.aviaSearchIsCompleted = new Component(
            browser,
            'avia-search-is-completed',
        );
    }

    async waitForSearchComplete(timeout = 2 * MINUTE): Promise<void> {
        await this.aviaSearchIsCompleted.waitForVisible(timeout);
    }

    async findVariantWithBadge(
        badgeName: string,
    ): Promise<TestAviaResultVariant> {
        const result = await this.variants.findSeries(variant => {
            return variant.checkBadgeExist(badgeName);
        });

        if (!result) {
            throw new Error('Отсутствуют искомые варианты');
        }

        return result;
    }
}

export class AviaSearchResultsDesktopPage extends AviaSearchResultsPage {
    readonly filters: AviaDesktopFilters;
    readonly resultsCount: Component;
    readonly nearestAirportsLink: Component;
    readonly dynamics: Component;
    readonly direct: Component;
    readonly subscriptionAndDynamicAsButtons: Component;
    readonly fog: TestAviaSearchResultFog;
    readonly noResultsYak: NoResultsYak;

    constructor(browser: WebdriverIO.Browser) {
        super(browser);

        this.filters = new AviaDesktopFilters(browser);
        this.resultsCount = new Component(browser, 'avia-variants-count');
        this.nearestAirportsLink = new Component(
            browser,
            'avia-nearest-airports-link',
        );
        this.dynamics = new Component(browser, 'avia-desktop-dynamics');
        this.direct = new Component(browser, 'avia-search-direct');
        this.subscriptionAndDynamicAsButtons = new Component(
            browser,
            'avia-search-subscriptionAndDynamicAsButtons',
        );
        this.fog = new TestAviaSearchResultFog(browser);
        this.noResultsYak = new NoResultsYak(browser);
    }
}
