import {delay} from 'helpers/project/common/delay';

export class ElementClass {
    selector?: string;

    readonly elementId: string;

    private readonly browser: WebdriverIO.Browser;
    private readonly element: WebDriverElement;

    constructor(browser: WebdriverIO.Browser, element: WebDriverElement) {
        this.browser = browser;
        this.element = element;

        // Support ElementClass as element
        if (element.elementId) {
            this.elementId = element.elementId;
        }
        // Support Webdriver node JSON as element
        else if (element.value) {
            this.elementId = element.value.ELEMENT;
        } else {
            const selector = element && element.selector;

            throw new Error(`Элемент не найден. Selector ${selector}`);
        }

        if (element.selector) {
            if (element.index) {
                this.selector = `${element.selector}:nth-child(${
                    element.index + 1
                })`;
            } else {
                this.selector = element.selector;
            }
        }
    }

    /** @deprecated - устаревший методы webdriver 4 */
    get(selector: string): Promise<ElementClass> | null {
        if (this.elementId) {
            return this.browser
                .elementIdElement(this.elementId, selector)
                .then(
                    (element: WebDriverElement) =>
                        new ElementClass(this.browser, element),
                );
        }

        return null;
    }

    async getHTML(): Promise<string | null> {
        if (this.element) {
            return this.element.getHTML(false);
        }

        return null;
    }

    async getText(): Promise<string> {
        if (this.element) {
            return this.element.getText();
        }

        return '';
    }

    async getValue(): Promise<string | null> {
        if (this.element) {
            return this.element.getAttribute('value');
        }

        return null;
    }

    async setValue(value: string | number): Promise<void | null> {
        if (this.element) {
            if (value) {
                return this.element.setValue(value);
            }

            return this.element.clearValue();
        }

        return null;
    }

    async addValue(value: string | number): Promise<void | null> {
        if (this.element && value) {
            return this.element.addValue(value);
        }

        return null;
    }

    async clearValue(withoutBackspace: boolean = false): Promise<void> {
        if (!withoutBackspace) {
            await this.tryClearWithBackspace();
        }

        await this.tryClearElement();
    }

    async click(): Promise<void> {
        if (this.element) {
            await this.element.click();
        }
    }

    /**
     * При использовании будь уверен, что this.selector корректный (индекс в nth-child),
     * т.к. в некоторых случаях он может формироваться неправильно
     */
    async clickJS(): Promise<void> {
        if (!this.selector) {
            return;
        }

        await this.browser.execute(selector => {
            const element = document.querySelector(selector);

            if (element instanceof HTMLElement) {
                element.click();
            } else if (element instanceof SVGElement) {
                const clickEvent = document.createEvent('SVGEvents');

                clickEvent.initEvent('click', true, true);
                element.dispatchEvent(clickEvent);
            }
        }, this.selector);

        // На прохождение анимаций + иногда браузер почему-то не мгновенно реагирует на js клик
        await delay(3000);
    }

    async getAttribute(attributeName: string): Promise<string | null> {
        if (this.element) {
            return this.element.getAttribute(attributeName);
        }

        return null;
    }

    /** @depricated - устаревший метод, используйте isDisplayed, isDisplayedInViewport, waitForDisplayed */
    async isVisible(): Promise<boolean> {
        if (this.selector) {
            return this.browser.isVisible(this.selector);
        }

        return false;
    }

    async isDisplayed(): Promise<boolean> {
        if (this.element) {
            return this.element.isDisplayed();
        }

        return false;
    }

    async isDisabled(): Promise<boolean> {
        if (this.element) {
            return (await this.element.getAttribute('disabled')) === 'true';
        }

        return false;
    }

    getLocation() {
        if (this.element) {
            return this.element.getLocation();
        }

        return null;
    }

    async getCssProperty(
        cssProperty: string,
    ): Promise<IWDIOCssProperty | null> {
        if (this.selector) {
            return await this.browser.getCssProperty(
                this.selector,
                cssProperty,
            );
        }

        return null;
    }

    async selectByAttribute(attribute: string, value: string): Promise<void> {
        if (this.element) {
            await this.element.selectByAttribute(attribute, value);
        }
    }

    async selectByIndex(index: number): Promise<void> {
        if (this.element) {
            await this.element.selectByIndex(index);
        }
    }

    private async tryClearWithBackspace(): Promise<void> {
        let val = await this.getValue();

        if (this.element) {
            while (val !== '') {
                // эмуляция нажатия backspace
                await this.element.addValue('\uE003');
                val = await this.getValue();
            }
        }
    }

    private async tryClearElement(): Promise<void> {
        if (this.element) {
            await this.element.clearValue();
        }
    }
}
