import {find} from 'p-iteration';

import prepareQaSelector from 'commands/lib/utils/qa';
import {prepareEndingQaSelector} from 'commands/lib/utils/endingQa';

import {ElementClass} from '../ElementClass/ElementClass';

export interface IConstructable<T extends Component> {
    new (browser: WebdriverIO.Browser, qa: QA): T;
}

type TQaPart = string | number | undefined | null;

export class Component implements IComponent {
    qa: string = '';

    protected readonly browser: WebdriverIO.Browser;

    protected _parentPath: string[] | undefined;

    constructor(browser: WebdriverIO.Browser, qa?: QA) {
        this.browser = browser;

        if (typeof qa === 'string') {
            this.qa = qa;
        } else if (typeof qa === 'object') {
            if (typeof qa.parent === 'object') {
                this.qa = this.joinQa(qa.key, qa.current);
                this._parentPath = qa.parent.path;
            } else {
                this.qa = this.joinQa(qa.key, qa.parent, qa.current);
                this._parentPath = qa.path;
            }
        }
    }

    get selector(): string {
        return this.prepareQaSelector(this.qa);
    }

    get path(): string[] {
        return this.getPathWithQa(this.qa);
    }

    get isTouch(): boolean {
        const chromeOptions =
            this.browser.requestedCapabilities['goog:chromeOptions'] || {};

        /**
         * Хак: сейчас у нас все мобильные устройства - это мобильная эмуляция на базе хром.
         * Для остальных эмуляций или реальных мобильных девайсов
         * это условие придется доработать
         */
        return Boolean(chromeOptions.mobileEmulation);
    }

    get isDesktop(): boolean {
        return !this.isTouch;
    }

    async getElement(
        selector: string,
        timeout?: number,
    ): Promise<ElementClass> {
        await this.waitForExist(selector, timeout);

        const elements: ElementClass[] = (await this.browser.$$(selector)).map(
            (e: WebDriverElement) => new ElementClass(this.browser, e),
        );

        if (elements.length > 1) {
            const element = await find(elements, e => e.isDisplayed());

            return element || elements[0];
        }

        return elements[0];
    }

    async getElementQA(qa?: string, timeout?: number): Promise<ElementClass> {
        return this.getElement(this.prepareQaSelector(qa || this.qa), timeout);
    }

    async getElements(
        selector: string,
        timeout?: number,
    ): Promise<ElementClass[]> {
        try {
            await this.waitForExist(selector, timeout);

            return (await this.browser.$$(selector)).map(
                (e: WebDriverElement) => new ElementClass(this.browser, e),
            );
        } catch (e) {
            return [];
        }
    }

    waitForExist(selector: string, timeout: number = 2000): Promise<void> {
        return this.browser.waitForExist(selector, timeout);
    }

    waitForVisible(timeout: number = 2000): Promise<void> {
        return this.browser.waitForVisible(
            this.prepareQaSelector(this.qa),
            timeout,
        );
    }

    async waitForHidden(timeout?: number): Promise<void> {
        return this.browser.waitForVisible(
            this.prepareQaSelector(this.qa),
            timeout,
            true,
        );
    }

    async waitUntil(
        conditionFunction: () => Promise<boolean>,
        timeout: number = 2000,
        timeoutMsg?: string,
    ): Promise<void> {
        await this.browser.waitUntil(
            async () => {
                return await conditionFunction();
            },
            {
                timeout,
                timeoutMsg,
            },
        );
    }

    async getText(timeout?: number): Promise<string> {
        const selector = this.prepareQaSelector(this.qa);
        const element = this.browser.$(selector);

        await element.waitForExist({timeout});

        return element.getText();
    }

    async getHTML(timeout?: number): Promise<string | null> {
        const selector = this.prepareQaSelector(this.qa);
        const element = this.browser.$(selector);

        await element.waitForExist({timeout});

        return element.getHTML(false);
    }

    /** @depricated - устаревший метод, используйте isDisplayed, isDisplayedInViewport, waitForDisplayed */
    async isVisible(timeout = 3000): Promise<boolean> {
        try {
            return (await this.getElementQA(undefined, timeout)).isVisible();
        } catch (e) {
            return false;
        }
    }

    async isDisplayed(timeout = 3000): Promise<boolean> {
        try {
            return (await this.getElementQA(undefined, timeout)).isDisplayed();
        } catch (e) {
            return false;
        }
    }

    async isDisplayedInViewport(): Promise<boolean> {
        const element = await this.browser.$(this.selector);

        return await element.isDisplayedInViewport();
    }

    async getAttribute(attributeName: string): Promise<string | null> {
        return (await this.getElementQA()).getAttribute(attributeName);
    }

    async isDisabled(): Promise<boolean | null> {
        return (await this.getElementQA()).isDisabled();
    }

    /**
     * Подскроливает к элементу, чтобы элемент оказался в центре viewport
     * @link https://webdriver.io/docs/api/element/scrollIntoView
     */
    async scrollIntoView(): Promise<void> {
        const element = await this.browser.$(this.selector);

        await element.scrollIntoView({block: 'center'});
    }

    async click(): Promise<void> {
        const e = await this.getElementQA();

        await e.click();
    }

    /**
     * Клик по элементу через JS
     * Не рекомендуется, так как не эмулирует настоящий клик
     * Только в редких кейсах, например работа во фрейме
     */
    async clickJS(): Promise<void> {
        const e = await this.getElementQA();

        await e.clickJS();
    }

    async getCssProperty(
        propertyName: string,
    ): Promise<IWDIOCssProperty | null> {
        return (await this.getElementQA()).getCssProperty(propertyName);
    }

    protected prepareQaSelector(qa: string): string {
        if (this._parentPath) {
            return this.prepareQaSelectorWithPath(qa);
        }

        return prepareQaSelector(qa);
    }

    protected prepareQaSelectorWithPath(qa?: string): string {
        return this.getPathWithQa(qa).map(prepareQaSelector).join(' ');
    }

    protected prepareEndingQaSelector(qa: string): string {
        if (this._parentPath) {
            const pathSelector = this.prepareQaSelectorWithPath();

            return `${pathSelector} ${prepareEndingQaSelector(qa)}`;
        }

        return prepareEndingQaSelector(qa);
    }

    protected getPathWithQa(qa?: string): string[] {
        const qaPart = qa ? [qa] : [];

        return (this._parentPath || []).concat(qaPart);
    }

    private joinQa(...parts: TQaPart[]): string {
        return parts.filter(key => key !== undefined && String(key)).join('-');
    }
}
