import {Component} from './Component';

export class Input extends Component {
    /**
     * @param value - значение для инпута
     * @param isMasked - есть ли маска у инпута. Если есть, то нужно вводить символы по одному иначе в некоторых браузерах могут ввестись не все символы.
     * @param clearBefore - признак необходимости очистки поля
     */
    async type(
        value: string,
        isMasked: boolean = false,
        clearBefore: boolean = true,
    ): Promise<void> {
        const elem = await this.getElementQA();
        const dividedValue = value.split('');

        if (clearBefore) {
            await elem.clearValue();
        }

        if (isMasked) {
            for (const char of dividedValue) {
                await elem.addValue(char);
            }
        } else {
            await elem.setValue(value);
        }
    }

    /**
     * Очищает поле ввода и вводитт в него новое значние
     *
     * @param isMasked - есть ли маска у инпута. Если есть, то нужно вводить символы по одному иначе в некоторых браузерах могут ввестись не все символы.
     */
    async setValue(
        value: string,
        options?: {isMasked?: boolean; withoutBackspace?: boolean},
    ): Promise<void> {
        await this.clearValue(options?.withoutBackspace);
        await this.type(value, options?.isMasked, false);
    }

    async getValue(): Promise<string | null> {
        const elem = await this.getElementQA();

        return elem.getValue();
    }

    async clearValue(withoutBackspace?: boolean): Promise<void> {
        const elem = await this.getElementQA();

        await elem.clearValue(withoutBackspace);
    }
}
