import {SECOND} from 'helpers/constants/dates';

import {SearchSuggestTrigger} from 'components/SearchSuggestTrigger';

import {Input} from './Input';
import {Component} from './Component';
import {DefaultSuggestItem} from './DefaultSuggestItem';
import {retry} from '../retry';
import {Button} from './Button';
import {ComponentArray} from './ComponentArray';
import {SearchFormFieldModal} from './SearchFormFieldModal';

export class SearchSuggest extends Component {
    _trigger: SearchSuggestTrigger;
    _modalTrigger: Component;
    _input: Input;
    _modalInput: Input;
    _modalResetButton: Button;
    suggestItems: ComponentArray<DefaultSuggestItem>;
    modal: SearchFormFieldModal;

    opened: boolean;

    constructor(browser: WebdriverIO.Browser, qa: QA = 'search-suggest') {
        super(browser, qa);

        this.opened = false;

        this.modal = new SearchFormFieldModal(browser);
        this._trigger = new SearchSuggestTrigger(browser, {
            parent: this.qa,
            current: 'trigger',
        });
        this._modalTrigger = new Component(browser, {
            parent: this.qa,
            current: 'modal-trigger',
        });

        this._input = new Input(browser, {
            parent: this.qa,
            current: 'trigger-input',
        });
        this._modalInput = new Input(browser, {
            parent: this.qa,
            current: 'modal-trigger-input',
        });

        this._modalResetButton = new Button(browser, {
            parent: this.qa,
            current: 'modal-trigger-reset',
        });

        this.suggestItems = new ComponentArray(
            browser,
            `${this.qa}-list-item`,
            DefaultSuggestItem,
        );
    }

    get trigger(): Component {
        if (!this.isTouch || !this.opened) {
            return this._trigger;
        }

        return this._modalTrigger;
    }

    get input(): Input {
        if (!this.isTouch || !this.opened) {
            return this._input;
        }

        return this._modalInput;
    }

    get resetButton(): Component {
        if (!this.isTouch || !this.opened) {
            return this._trigger.reset;
        }

        return this._modalResetButton;
    }

    async click(): Promise<void> {
        await this.trigger.click();

        this.opened = true;
    }

    async setValue(value: string, pickFirst?: boolean): Promise<void> {
        await retry(
            async () => {
                await this.resetValue();

                await this.setInputValue(value);

                await this.selectSuggestItem(value, pickFirst);
            },
            {delay: 500, attempts: 10},
        )();
    }

    async getTriggerValue(): Promise<string> {
        return this._trigger.title.getText();
    }

    async getValue(): Promise<string | null> {
        return await this.input.getValue();
    }

    // Метод закрытия саджеста по кнопке "<-" (только тач)
    async cancel(): Promise<void> {
        await this.modal.cancel.click();
        this.opened = false;
    }

    async resetValue(): Promise<void> {
        const value = await this.input.getValue();

        if (!value) {
            return;
        }

        await this.click();

        /**
         * Нужно начать вводить символы, чтобы появился крестик очистки
         */
        if (this.isTouch) {
            await this.input.type('а');
        }

        await this.resetButton.click();
    }

    async setInputValue(value: string): Promise<void> {
        await this.input.setValue(value);

        /**
         * Ждем пока пересчиается выпадающий список
         */
        await this.browser.pause(SECOND);
    }

    private async selectSuggestItem(
        searchTitle: string,
        pickFirst?: boolean,
    ): Promise<void> {
        const suggestItem = await this.suggestItems.findSeries(async item => {
            const title = await item.title.getText();

            if (pickFirst) {
                return true;
            }

            return searchTitle.includes(title);
        });

        if (!suggestItem) {
            throw new Error(`В suggest не найден "${searchTitle}"`);
        }

        await suggestItem.click();

        this.opened = false;
    }
}
