import {
    TUSClient,
    CreatePortalAccountResult,
    GetAccountResult,
} from '@yandex-int/tus-client';

import {MINUTE, SECOND} from 'helpers/constants/dates';

export type TCreateAccountResult = CreatePortalAccountResult['account'];
export type TGetAccountResult = GetAccountResult['account'];

export enum EAccountTag {
    ALL = 'all',
}

export default class Account {
    client: TUSClient;

    constructor() {
        const tusOAuthToken = process.env.E2E_OAUTH_TOKEN;

        if (!tusOAuthToken) {
            throw new Error(
                'process.env.E2E_OAUTH_TOKEN is undefined. See .env files',
            );
        }

        this.client = new TUSClient(tusOAuthToken, {
            tus_consumer: 'ya-travel-frontend',
            env: 'test',
        });
    }

    async create(tags = EAccountTag.ALL): Promise<TCreateAccountResult> {
        const res: CreatePortalAccountResult =
            await this.client.createPortalAccount({tags});

        return res.account;
    }

    async remove(id: string): Promise<void> {
        try {
            await this.client.removeAccountFromTUS({uid: id});
        } catch {}
    }

    /**
     * Берет свободные учетные записи из TUS
     * или при неудаче создает новые
     */
    async getOrCreate(): Promise<{
        isNew: boolean;
        account: TCreateAccountResult | TGetAccountResult;
    }> {
        try {
            const res: GetAccountResult = await this.client.getAccount({
                tags: EAccountTag.ALL,
                ignore_locks: false,
                /**
                 * В секундах
                 */
                lock_duration: (10 * MINUTE) / SECOND,
            });

            return {
                isNew: false,
                account: res.account,
            };
        } catch (e) {
            console.error(e);

            const account = await this.create();

            return {isNew: true, account};
        }
    }
}
