import moment, {Moment} from 'moment';

import {AdFoxBanner} from 'helpers/project/common/components/AdFoxBanner';
import dateFormats from 'helpers/utilities/date/formats';

import {Button} from 'components/Button';
import {Component} from 'components/Component';
import {DatePicker} from 'components/DatePicker';
import {SearchSuggest} from 'components/SearchSuggest';

import {HotelsTravellers} from './HotelsTravellers';

interface IFillHotelsSearchFormParams {
    adults?: number;
    childrenAges?: number[];
    checkinDate: string;
    checkoutDate: string;
    place?: string;
}

export class HotelsSearchForm extends Component {
    place: SearchSuggest;
    period: DatePicker;
    travellers: HotelsTravellers;
    submitButton: Button;

    _adFox: AdFoxBanner;

    constructor(browser: WebdriverIO.Browser, qa?: QA) {
        super(browser, qa);

        this.place = new SearchSuggest(this.browser, {
            parent: this.qa,
            current: 'search-suggest-to',
        });

        this.period = new DatePicker(this.browser, {
            parent: this.qa,
            current: 'date-picker',
        });
        this.travellers = new HotelsTravellers(this.browser, {
            parent: this.qa,
            current: 'travellers',
        });

        this.submitButton = new Button(this.browser, {
            parent: this.qa,
            current: 'submit',
        });

        this._adFox = new AdFoxBanner(this.browser);
    }

    async submitForm(): Promise<void> {
        await this.submitButton.click();
    }

    async fill({
        adults = 2,
        childrenAges = [],
        checkinDate,
        checkoutDate,
        place,
    }: IFillHotelsSearchFormParams): Promise<void> {
        await this._adFox.disableEvents();

        if (place) {
            await this.place.setValue(place);
        }

        await this.period.selectStartDate(checkinDate);
        await this.period.selectEndDate(checkoutDate);

        await this.travellers.click();
        await this.travellers.adultsCount.setCount(adults);

        for (const age of childrenAges) {
            await this.travellers.addChildren(age);
        }

        if (this.isTouch) {
            await this.travellers.modal.clickCompleteButton();
        }

        await this._adFox.enableEvents();
    }

    async getDates(): Promise<{startDate: Moment; endDate: Moment}> {
        const startDate = await this.period.startTrigger.value.getText();
        const endDate = await this.period.endTrigger.value.getText();

        return {
            startDate: moment(startDate, dateFormats.HUMAN_SHORT),
            endDate: moment(endDate, dateFormats.HUMAN_SHORT),
        };
    }
}
