import moment, {Moment} from 'moment';

import extractNumbers from 'helpers/utilities/extractNumbers';
import dateFormats from 'helpers/utilities/date/formats';

interface IParseDatesPeriodOptions {
    separator?: string;
    dateFormat?: string;
    monthFormat?: string;
    yearFormat?: string;
}

/**
 * Парсит строку с периодом формата
 * 4 янв или 4 янв'23 или 4 дек – 1 янв'23
 */
export default function parseDatesPeriod(
    dates: string,
    options: IParseDatesPeriodOptions = {},
): {
    startDate: Moment;
    endDate: Moment;
} {
    const {
        separator = '–',
        dateFormat = dateFormats.DATE,
        monthFormat = dateFormats.SHORT_MONTH,
        yearFormat = dateFormats.SHORT_YEAR,
    } = options;

    const [start, end] = dates.split(separator).map(s => s.trim());

    if (!start || !end) {
        throw new Error(`Период дат не содержит разделитель: ${separator}`);
    }

    const parsedStart = parseDate(start);
    const parsedEnd = parseDate(end);

    const format = `${dateFormat} ${monthFormat}'${yearFormat}`;

    return {
        startDate: moment(
            formatDate(parsedStart, parsedEnd, yearFormat),
            format,
        ),
        endDate: moment(formatDate(parsedEnd, parsedEnd, yearFormat), format),
    };
}

interface IParsedDate {
    day: number;
    month?: string;
    year?: number;
}

function formatDate(
    date: IParsedDate,
    additional: IParsedDate,
    yearFormat: string,
): string {
    return `${date.day} ${date.month || additional.month}'${
        date.year || additional.year || moment().format(yearFormat)
    }`;
}

function parseDate(str: string): {
    day: number;
    month?: string;
    year?: number;
} {
    const [day, year] = extractNumbers(str);

    const match = str.match(/\d+\s([А-Яа-яЁё]+)/);
    const month = match?.[1];

    return {day, month, year};
}
