const fs = require('fs-extra');

class SkipTestsByReport {
    constructor({hermione, reportFile}) {
        this.hermione = hermione;
        this.reportFile = reportFile;
        this.skippedTests = [];
    }

    start() {
        this.initBindings();
    }

    initBindings() {
        this.hermione.on(
            this.hermione.events.INIT,
            this.handleHermioneInit.bind(this),
        );
        this.hermione.on(
            this.hermione.events.AFTER_TESTS_READ,
            this.handleHermioneAfterTestsRead.bind(this),
        );
    }

    /* Handlers */

    async handleHermioneInit() {
        return await this.readReportFile();
    }

    handleHermioneAfterTestsRead(testCollection) {
        this.startSkipTests(testCollection);
    }

    /* Helpers */

    getTestIdentifier(fullTitle, browserId) {
        if (browserId) {
            return `${fullTitle}-${browserId}`;
        }

        return fullTitle;
    }

    getSkippedTestsMap() {
        return this.skippedTests.reduce((skippedTestsMap, skippedTest) => {
            const {fullTitle, browserId} = skippedTest;
            const testIdentifier = this.getTestIdentifier(fullTitle, browserId);

            skippedTestsMap[testIdentifier] = skippedTest;

            return skippedTestsMap;
        }, {});
    }

    checkUnknownSkippedTest(skippedTestsMap) {
        Object.keys(skippedTestsMap).forEach(skippedTestIdentifier => {
            if (!skippedTestsMap[skippedTestIdentifier].passed) {
                throw new Error(
                    `Unknown skipped test: ${skippedTestIdentifier}, plugin: 'SkipTestsByReport'`,
                );
            }
        });
    }

    /* Actions */

    async readReportFile() {
        try {
            this.skippedTests = await fs.readJson(this.reportFile);
        } catch (e) {
            throw new Error(
                `Failed to read ${this.reportFile}, plugin: 'SkipTestsByReport'`,
            );
        }
    }

    startSkipTests(testCollection) {
        if (this.skippedTests.length) {
            const skippedTestsMap = this.getSkippedTestsMap();

            testCollection.eachTest(test => {
                const {browserId} = test;
                const fullTitle = test.fullTitle();
                const testIdentifier = this.getTestIdentifier(
                    fullTitle,
                    browserId,
                );
                const skippedTest =
                    skippedTestsMap[fullTitle] ||
                    skippedTestsMap[testIdentifier];

                if (skippedTest) {
                    const {reason} = skippedTest;
                    const skippedTestIdentifier = skippedTestsMap[fullTitle]
                        ? fullTitle
                        : testIdentifier;

                    skippedTestsMap[skippedTestIdentifier].passed = true;

                    test.pending = true;
                    test.skipReason = reason;
                }
            });

            this.checkUnknownSkippedTest(skippedTestsMap);
        }
    }
}

module.exports = SkipTestsByReport;
