/**
 * @typedef {import('./types/TPushClientStatusesInfo').TPushClientStatusesInfo} TPushClientStatusesInfo
 * @typedef {import('./types/IPushClientSensors').IPushClientSensors} IPushClientSensors
 */

const {execSync} = require('child_process');
const path = require('path');
const {Registry} = require('@yandex-data-ui/sensors');

const {FILE} = require('./types/pushClientStatusTypes');

/**
 * @returns {IPushClientSensors}
 */
async function getPushClientSensors() {
    const statusesInfo = getPushClientStatusesInfo();

    const requiredSensors = {
        'commit-delay': 'commit delay',
        lag: 'lag',
    };

    const solomonRegistry = new Registry();

    for (const info of statusesInfo) {
        if (info.type === FILE) {
            const {name} = path.parse(info.name);

            for (const [sensorId, sensorName] of Object.entries(
                requiredSensors,
            )) {
                solomonRegistry
                    .gauge('push-client', {
                        log_name: name,
                        type: sensorId,
                    })
                    .addValue(info[sensorName]);
            }

            const ts = Math.floor(Number(new Date()) / 1000);
            const lastSendTime = info['last send time'];
            const timeDelay = ts - lastSendTime;

            solomonRegistry
                .gauge('push-client', {
                    log_name: name,
                    type: 'time-delay',
                })
                .addValue(timeDelay);
        }
    }

    return solomonRegistry.getData();
}

/**
 * @returns {TPushClientStatusesInfo}
 */
function getPushClientStatusesInfo() {
    const cmdOut = execSync(getPushClientStatusCommand(process.env.APP_ENV));
    return JSON.parse(cmdOut.toString());
}

function getPushClientStatusCommand(env) {
    return `/usr/bin/timeout -k 10 -s 9 10 /usr/bin/push-client -f -c ${getPushClientConfigFile(
        env,
    )} --status --json`;
}

function getPushClientConfigFile(env) {
    return `/etc/statbox-push-client/${env}/push-client.yaml`;
}

module.exports = getPushClientSensors;
