import _getFp from 'lodash/fp/get';
import {ParsedQuery} from 'query-string';
import {asClass, asValue, Resolver} from 'awilix';
import {Span} from 'opentracing';

import {ECrowdTestHeaders, EXredirectHeaders} from 'server/constants/headers';

import {
    Request,
    Response,
    ICookies,
    BlackboxInfo,
    CoreTelemetrySendStats,
    CoreConfig,
    CoreUtils,
} from '@yandex-data-ui/core/lib/types';
import {IUserSplit, TRawExperiments} from './providers/experiments/types';
import {IUATraits} from 'types/common/IUATraits';
import {TBlackboxUserInfo} from 'types/common/userInfo/TBlackboxUserInfo';
import IAttributionData from 'server/utilities/DataStorage/AttributionData/types/IAttributionData';
import IUserData from 'server/utilities/DataStorage/UserData/types/IUserData';
import {TSrcParams} from 'types/TSourceParams';
import {EApiEntry} from 'types/EApiEntry';
import {IGeobaseProvider} from 'server/providers/geoBase/types';

import {
    ILogger,
    TLogger,
    TErrorLogger,
    AppLogger,
} from 'server/utilities/Logger';
import {CurrencyType} from 'utilities/currency/CurrencyType';
import {IHttpClient} from 'server/utilities/HttpClient/IHttpClient';
import {HttpClient} from 'server/utilities/HttpClient/HttpClient';
import {AviaGatewayBackendApiClient} from 'server/utilities/AviaBackendApiClient/AviaGatewayBackendApiClient';
import {IAviaBackendApiClient} from 'server/utilities/AviaBackendApiClient/IAviaBackendApiClient';
import {AviaBackendRestApiClient} from 'server/utilities/AviaBackendApiClient/AviaBackendRestApiClient';
import {
    getFileLoggerWrapperGetter,
    TFileLoggerWrapperGetter,
} from './loggers/utils/getFileLoggerWrapper';
import {getUserSplitStandalone} from './providers/experiments/utils/getUserSplitStandalone';
import {AviaBackendApiClient} from 'server/utilities/AviaBackendApiClient/AviaBackendApiClient';
import AttributionData from 'server/utilities/DataStorage/AttributionData/AttributionData';
import UserData from 'server/utilities/DataStorage/UserData/UserData';
import {parseSrcParams} from 'server/utilities/srcParams/parseSrcParams';
import {HotelsMetaParamsBuilder} from 'server/utilities/hotels/HotelsMetaParamsBuilder/HotelsMetaParamsBuilder';

import {TinyUrlApi} from 'server/api/TinyUrl/TinyUrlApi';
import {TinyUrlService} from 'server/services/TinyUrlService/TinyUrlService';
import {ImageGeneratorApi} from 'server/api/ImageGeneratorApi/ImageGeneratorApi';
import {ImageGeneratorService} from 'server/services/ImageGeneratorService/ImageGeneratorService';
import {NotifierApi} from 'server/api/NotifierApi/NotifierApi';
import {NotifierService} from 'server/services/NotifierService/NotifierService';
import {ExperimentsService} from 'server/services/ExperimentsService/ExperimentsService';
import {CurrenciesService} from 'server/services/CurrenciesService/CurrenciesService';
import {GeobaseApi} from 'server/api/GeobaseApi/GeobaseApi';
import {GeobaseService} from 'server/services/GeobaseService/GeobaseService';
import LocalGeobase from 'server/providers/geoBase/LocalGeobase';
import HTTPGeobase from 'server/providers/geoBase/HttpGeobase';
import GeobaseProvider from 'server/providers/geoBase/GeobaseProvider';

function createLogger(
    log: TLogger,
    logWarn: TErrorLogger,
    logError: TErrorLogger,
    extra?: any,
): ILogger {
    return new AppLogger({log, logWarn, logError, extra});
}

function createApiHostResolver(hostsDict: Record<EApiEntry, string>) {
    return (apiName: EApiEntry): string => hostsDict[apiName];
}

function createSrcParamsResolver(srcParams: TSrcParams) {
    return (apiName: EApiEntry): Record<string, string> | undefined =>
        srcParams[apiName];
}

interface ITVMTicket {
    error?: string;
    ticket?: string;
    // eslint-disable-next-line camelcase
    tvm_id: number;
}

export interface INationalSettings {
    nationalCurrency: CurrencyType;
    allowedCurrencies: CurrencyType[];
}

const useLocalGeobase = Boolean(process.env.USE_LOCAL_GEOBASE);

function createServiceTicketResolver(tickets: Record<string, ITVMTicket>) {
    return (serviceName: string): string => {
        let name = serviceName;

        // TODO: Нужно для плавного перехода к новой переменной в tvm. hotelsAPI нужно превратить в travelAPI, потому что этот api расширяется до общепортального.
        if (name === 'travelAPI') {
            name = 'hotelsAPI';
        }

        const serviceTVM = tickets[name];

        if (!serviceTVM) {
            throw new Error('No ticket for service: ' + name);
        }

        if (!serviceTVM.ticket) {
            throw new Error(
                serviceTVM.error || 'No ticket for service: ' + name,
            );
        }

        return serviceTVM.ticket;
    };
}

function createSendClickHouseStats(
    stats: CoreTelemetrySendStats,
    table: string,
): ((data: {[name: string]: string | number}) => void) | undefined {
    if (!table) {
        return undefined;
    }

    return (data: {[name: string]: string | number}): void => {
        return stats(table, data);
    };
}

const createAppConfigResolver = (utils: CoreUtils): CoreConfig =>
    _getFp<CoreUtils, 'config'>(['config'])(utils);

export default function getContainerConfig(
    req: Request,
    res: Response,
    {isSSR}: {isSSR: boolean},
): TResolverCollection<IDependencies> {
    const {
        utils,
        blackbox,
        cookies,
        query,
        userInfo,
        session,
        tvm,
        id,
        tld,
        hostname,
        userSplit,
        uatraits,
        ip,
        headers,
        rootSpan,
    } = req;

    if (!tvm || !tvm.tickets) {
        throw new Error('TVM is not defined');
    }

    if (!session.yaTravelSessionUid) {
        throw new Error('Session UID is not defined');
    }

    const {log, logWarn, logError, config, stats} = utils;

    const yandexuid = cookies.yandexuid;

    const nationalSettings =
        config.nationalSettings[tld] || config.nationalSettings.default;

    const apiHostResolver = createApiHostResolver(config.servicesAPI);
    const serviceTicketResolver = createServiceTicketResolver(tvm.tickets);
    const appConfigResolver = createAppConfigResolver(utils);
    const srcParamsResolver = createSrcParamsResolver(
        userInfo?.isYandexNetwork ? parseSrcParams(query) : {},
    );

    const logger = createLogger(log, logWarn, logError, {
        yandexuid,
        login: userInfo && 'login' in userInfo && userInfo.login,
    });

    const fileLoggerWrapperGetter = getFileLoggerWrapperGetter(
        log,
        logWarn,
        logError,
        utils.config,
    );

    const sendClickHouseStats = createSendClickHouseStats(
        stats,
        config.telemetryClickHouseEnvTable,
    );

    const setCookie = res.cookie.bind(res);

    const getUserSplit = (): Promise<IUserSplit> => getUserSplitStandalone(req);

    const referrer = req.get('referrer');

    const testBuckets = userSplit && userSplit.boxes;

    const attributionData = new AttributionData({
        logger,
        query,
        cookies,
        setCookie,
        referrer,
        testBuckets,
    });

    attributionData.updateCookiesAndFillData();

    const crowdTestHostHeader = headers[ECrowdTestHeaders.X_YA_CROWD_TEST_HOST];

    const geobaseProvider = new GeobaseProvider(
        useLocalGeobase ? new LocalGeobase() : new HTTPGeobase(),
        logger,
    );

    return {
        rootSpan: asValue(rootSpan),
        logger: asValue(logger),
        blackbox: asValue(blackbox),
        uatraits: asValue(uatraits),
        userInfo: asValue(userInfo),
        userAgent: asValue(headers['user-agent']),
        isSSR: asValue(isSSR),
        isBot: asValue(req.isBot),
        ip: asValue(headers['x-real-ip'] as string),
        forwardedFor: asValue(headers['x-forwarded-for'] as string),
        isCrowdTestProxy: asValue(
            Boolean(headers[ECrowdTestHeaders.X_YA_CROWD_TEST_PROXY]),
        ),
        crowdTestHost: asValue(
            typeof crowdTestHostHeader === 'string'
                ? crowdTestHostHeader
                : undefined,
        ),
        isFromXredirect: asValue(
            headers[EXredirectHeaders.X_YA_TRAVEL_IS_FROM_XREDIRECT] === '1',
        ),
        query: asValue(query),
        cookies: asValue(cookies),
        referrer: asValue(referrer),
        setCookie: asValue(setCookie),
        sendClickHouseStats: asValue(sendClickHouseStats),
        yandexuid: asValue<string>(cookies.yandexuid),
        yandexGid: asValue<string>(cookies.yandex_gid),
        nationalVersion: asValue<string>(req.nationalVersion),
        nationalSettings: asValue(nationalSettings),
        preferredCurrency: asValue(
            cookies['preferred_currency'] as CurrencyType | undefined,
        ),
        isE2E: asValue(cookies.e2e_test === 'true'),
        mockTrips: asValue(cookies['MOCK_TRIPS_COOKIE_NAME'] === 'true'),

        sessionKey: asValue<string>(session.yaTravelSessionUid),
        appConfig: asValue<CoreConfig>(appConfigResolver),
        requestHostname: asValue<string>(hostname),
        requestId: asValue(id),
        requestIp: asValue<string>(ip),
        rawExperiments: asValue(userSplit?.rawExperiments),
        testBuckets: asValue(testBuckets),
        getApiHost: asValue(apiHostResolver),
        getSrcParams: asValue(srcParamsResolver),
        getServiceTicket: asValue(serviceTicketResolver),
        attributionData: asValue(attributionData),
        tinyUrlApi: asClass(TinyUrlApi),
        tinyUrlService: asClass(TinyUrlService),

        userData: asClass(UserData).scoped(),
        fileLoggerWrapperGetter: asValue(fileLoggerWrapperGetter),

        httpClient: asClass(HttpClient),
        aviaGatewayBackendApiClient: asClass(AviaGatewayBackendApiClient),
        aviaBackendApiClient: asClass(AviaBackendApiClient),
        aviaBackendRestApiClient: asClass(AviaBackendRestApiClient),

        hotelsMetaParamsBuilder: asClass(HotelsMetaParamsBuilder),

        getUserSplit: asValue(getUserSplit),

        geobaseApi: asClass(GeobaseApi),
        geobaseProvider: asValue(geobaseProvider),
        geobaseService: asClass(GeobaseService),

        currenciesService: asClass(CurrenciesService),

        experimentsService: asClass(ExperimentsService),

        notifierApi: asClass(NotifierApi),
        notifierService: asClass(NotifierService),

        imageGeneratorApi: asClass(ImageGeneratorApi),
        imageGeneratorService: asClass(ImageGeneratorService),
    };
}

type TResolverCollection<Resolvers extends Record<string, any>> = {
    [P in keyof Resolvers]: Resolver<Resolvers[P]>;
};

export interface IDependencies {
    logger: ILogger;
    httpClient: IHttpClient;
    blackbox: BlackboxInfo;
    uatraits: IUATraits;
    sendClickHouseStats?: (data: {[name: string]: string | number}) => void;
    userInfo: TBlackboxUserInfo;
    userAgent?: string;
    ip: string;
    forwardedFor: string;
    query: ParsedQuery;
    cookies: ICookies;
    setCookie: Response['cookie'];
    referrer: string | undefined;
    isSSR: boolean;
    isBot: boolean;
    isCrowdTestProxy: boolean;
    crowdTestHost: string | undefined;
    /** Признак, что запрос пришел с домена типа xredirect.yandex.ru */
    isFromXredirect: boolean;
    yandexuid: string;
    yandexGid?: string;
    preferredCurrency: CurrencyType | undefined;
    sessionKey: string;
    appConfig: CoreConfig;
    requestHostname: string;
    requestId: string;
    requestIp: string;
    nationalVersion: string;
    nationalSettings: INationalSettings;
    rawExperiments?: TRawExperiments;
    testBuckets?: string;
    getApiHost: (apiName: EApiEntry) => string;
    getSrcParams: (apiName: EApiEntry) => Record<string, string> | undefined;
    getServiceTicket: (apiName: string) => string;
    attributionData: IAttributionData;
    userData: IUserData;
    rootSpan?: Span;
    isE2E: boolean;
    mockTrips: boolean;

    fileLoggerWrapperGetter: TFileLoggerWrapperGetter;
    aviaGatewayBackendApiClient: IAviaBackendApiClient;
    aviaBackendApiClient: IAviaBackendApiClient;
    aviaBackendRestApiClient: IAviaBackendApiClient;
    tinyUrlApi: TinyUrlApi;
    tinyUrlService: TinyUrlService;

    hotelsMetaParamsBuilder: HotelsMetaParamsBuilder;

    getUserSplit: () => Promise<IUserSplit>;

    geobaseApi: GeobaseApi;
    geobaseProvider: IGeobaseProvider;
    geobaseService: GeobaseService;

    currenciesService: CurrenciesService;

    experimentsService: ExperimentsService;

    notifierApi: NotifierApi;
    notifierService: NotifierService;

    imageGeneratorApi: ImageGeneratorApi;
    imageGeneratorService: ImageGeneratorService;
}
