/**
 * @typedef {import('@yandex-data-ui/core/lib/types').Request} Request
 */

const SECURE_METHODS = ['POST', 'PUT', 'DELETE', 'PATCH'];

const isSecureMethod = method => SECURE_METHODS.includes(method.toUpperCase());

/**
 * Проверяем, что источник и приёмник запроса - это один и тот же ресурс
 * @param {Request} req
 * @return {boolean}
 */
const isInternalRequest = req => {
    const source = req.headers.origin || req.headers.referer;
    const targetHostname = req.hostname;
    const sourceHostname = source && new URL(source).hostname;

    if (sourceHostname && sourceHostname === targetHostname) {
        return true;
    }

    req.utils.logWarn('CSRF attack: it`s not internal request', req);

    return false;
};

/**
 * Проверяем наличие кастомного заголовка
 * @see https://cheatsheetseries.owasp.org/cheatsheets/Cross-Site_Request_Forgery_Prevention_Cheat_Sheet.html#use-of-custom-request-headers
 * @param {Request} req
 * @return {boolean}
 */
const hasCustomHeader = req => {
    if ('x-csrf-token' in req.headers) {
        return true;
    }

    req.utils.logWarn('CSRF attack: request without custom header', req);

    return false;
};

const isAllowedRequest = req => isInternalRequest(req) && hasCustomHeader(req);

module.exports = (req, res, next) => {
    if (isSecureMethod(req.method) && !isAllowedRequest(req)) {
        /*
        res.status(419).send({
            error: error.message,
        });

        return;
        */
    }

    next();
};
