const {globalTracer, Tags} = require('opentracing');
const {get} = require('lodash');

const {JAEGER_TRACER_OPTIONS} = require('../constants/common');

const {PAGE_SERVICE_NAME, TRACER_DEBUG_TAG_NAME} = JAEGER_TRACER_OPTIONS;

module.exports = serviceName => (req, res, next) => {
    try {
        const tracer = globalTracer();
        // Этот тэг нужено выставлять первым при генерации Span,
        // т.к. если сам Span не попадет в трассировку по вероятностной характеристике,
        // то многие последующие операции с ним будут игнорироваться
        const rootSamplingTags = getSamplingTags(req);
        const operationName =
            serviceName === PAGE_SERVICE_NAME ? '' : ` - ${req.path}`;

        // Встраиваем в запрос root спан, чтобы в дальнейших запросах к бэкам указывать его как родительский
        req.rootSpan = tracer.startSpan(`${serviceName}${operationName}`, {
            tags: rootSamplingTags,
        });
        setSpanTags(req);

        res.once('finish', function () {
            req.rootSpan.finish();
        });
    } catch (error) {
        req.utils.logError('JAEGER_TRACER_ERROR');
    }

    next();
};

function getSamplingTags(req) {
    const configForceDebugTag = get(
        req,
        'utils.config.travelTracing.meta.forceDebugTag',
    );
    const forceDebugTag = configForceDebugTag || get(req, 'userInfo.isStaff');

    return forceDebugTag ? {[Tags.SAMPLING_PRIORITY]: 1} : {};
}

function setSpanTags(req) {
    const requestId = req.headers && req.headers['x-request-id'];
    const jaegerDebugId = req.headers && req.headers[TRACER_DEBUG_TAG_NAME];

    req.rootSpan.addTags({
        path: req.path,
        query: req.query,
        'X-Request-Id': requestId,
    });

    if (jaegerDebugId) {
        req.rootSpan.setTag(TRACER_DEBUG_TAG_NAME, jaegerDebugId);
        req.rootSpan.setBaggageItem(TRACER_DEBUG_TAG_NAME, jaegerDebugId);
    }
}
