import {Action} from 'typesafe-actions';

import {
    INITIAL_TOGGLERS_STORE,
    STORED_TOGGLERS_COOKIE_KEY,
    TOGGLERS_EXPIRES_MS,
} from 'constants/togglers';

import {Request, Response} from '@yandex-data-ui/core/lib/types';
import {EToggler} from 'types/common/togglers/EToggler';
import {TCommonTogglersStore} from 'types/common/togglers/TCommonTogglersStore';

import {setTogglers} from 'reducers/common/togglers/actions';

const TOGGLERS_KEYS = Object.values(EToggler);

export default function fetchTogglers(req: Request, res: Response): Action {
    const storedTogglers = req.cookies[STORED_TOGGLERS_COOKIE_KEY];

    try {
        const parsedTogglers: PartialRecord<EToggler, boolean> =
            JSON.parse(storedTogglers);

        let needToUpdateCookie = false;

        /**
         * У старых пользователей не будет в куках храниться значение новых переключателей из EToggler.
         * Нужно обрабатывать такие случаи и выставлять значение по умолчанию.
         */
        const validTogglers = TOGGLERS_KEYS.reduce<TCommonTogglersStore>(
            (accTogglers, togglerKey) => {
                const parsedToggler = parsedTogglers[togglerKey];

                accTogglers[togglerKey] =
                    parsedTogglers[togglerKey] ??
                    INITIAL_TOGGLERS_STORE[togglerKey];

                if (!parsedToggler) {
                    needToUpdateCookie = true;
                }

                return accTogglers;
            },
            {} as TCommonTogglersStore,
        );

        if (needToUpdateCookie) {
            res.cookie(
                STORED_TOGGLERS_COOKIE_KEY,
                JSON.stringify(validTogglers),
                {
                    maxAge: TOGGLERS_EXPIRES_MS,
                },
            );
        }

        return setTogglers(validTogglers);
    } catch (err) {
        res.cookie(
            STORED_TOGGLERS_COOKIE_KEY,
            JSON.stringify(INITIAL_TOGGLERS_STORE),
            {
                maxAge: TOGGLERS_EXPIRES_MS,
            },
        );

        return setTogglers(INITIAL_TOGGLERS_STORE);
    }
}
