import React from 'react';
import ReactHelmet from 'react-helmet';
import htmlEscape from 'htmlescape';
import classnames from 'classnames';
import serialize from 'serialize-javascript';
import {ParsedQuery} from 'query-string';

import {METRIKA_COUNTER_ID} from 'constants/common';
import {IOS} from 'utilities/deviceType/constants';

import {EToggler} from 'types/common/togglers/EToggler';

import {StoreInterface} from 'reducers/storeTypes';

import {
    getUserGeoLocation,
    getUserInfo,
} from 'selectors/common/userInfoSelector';
import userSplitSelector from 'selectors/common/userSplitSelector';
import togglersSelector from 'selectors/common/togglers/togglersSelector';

import {isOS} from 'utilities/deviceType/isOS';

import Rum from 'components/Rum/Rum';

import YandexErrors from '../YandexErrors/YandexErrors';

interface IHtmlContainerProps {
    appVersion: string;
    appString: string;
    preloadScriptElements: React.ReactElement[];
    scriptElements: React.ReactElement[];
    reduxState: StoreInterface;
    criticalCss: string;
    reqPath: string;
    reqQuery: ParsedQuery;
    nonce: string;
    webvisor: boolean;
    appData: IAppData;
}

export default class HtmlContainer extends React.Component<IHtmlContainerProps> {
    static renderHelmetStaticAndMeta(): React.ReactNode {
        const helmetInstance = ReactHelmet.renderStatic();

        return (
            <React.Fragment>
                <meta charSet="utf-8" />
                <meta
                    name="viewport"
                    content="width=device-width, initial-scale=1, maximum-scale=1, user-scalable=no"
                />
                {/* https://developer.apple.com/library/archive/documentation/AppleApplications/Reference/SafariHTMLRef/Articles/MetaTags.html#//apple_ref/doc/uid/TP40008193-SW1 */}
                <meta name="format-detection" content="telephone=no" />
                {helmetInstance.base.toComponent()}
                {helmetInstance.title.toComponent()}
                {helmetInstance.meta.toComponent()}
                {helmetInstance.link.toComponent()}
                {helmetInstance.script.toComponent()}
            </React.Fragment>
        );
    }

    private static renderPreloadFonts(
        wasFontsLoadedBefore: boolean,
    ): React.ReactNode {
        const preloadType = wasFontsLoadedBefore ? 'preload' : 'prefetch';

        return (
            <>
                <link
                    rel={preloadType}
                    as="font"
                    type="font/woff2"
                    href="//yastat.net/s3/home/fonts/ys/1/text-regular.woff2"
                    crossOrigin="anonymous"
                />
                <link
                    rel={preloadType}
                    as="font"
                    type="font/woff2"
                    href="//yastat.net/s3/home/fonts/ys/1/text-medium.woff2"
                    crossOrigin="anonymous"
                />
                <link
                    rel={preloadType}
                    as="font"
                    type="font/woff2"
                    href="//yastat.net/s3/home/fonts/ys/1/text-bold.woff2"
                    crossOrigin="anonymous"
                />
            </>
        );
    }

    static defaultProps = {
        scriptElements: [],
        preloadScriptElements: [],
        criticalCss: '',
        reduxState: {},
    };

    private renderCriticalCss(): React.ReactNode {
        const {criticalCss} = this.props;

        return (
            <style
                type="text/css"
                dangerouslySetInnerHTML={{
                    __html: criticalCss,
                }}
            />
        );
    }

    private renderScriptWithReduxState(): React.ReactNode {
        const {reduxState, nonce} = this.props;
        const preLoadedReduxState = `window.__PRELOADED_REDUX_STATE__ = ${htmlEscape(
            reduxState,
        )}`;

        return (
            <script
                nonce={nonce}
                dangerouslySetInnerHTML={{__html: preLoadedReduxState}}
            />
        );
    }

    private getChunksScripts(): React.ReactElement[] {
        return this.props.scriptElements;
    }

    private getPreloadScripts(): React.ReactElement[] {
        return this.props.preloadScriptElements;
    }

    private getRootNode(): React.ReactNode {
        const {appString} = this.props;

        return <div id="app" dangerouslySetInnerHTML={{__html: appString}} />;
    }

    private renderMetrikaCounter(): React.ReactNode {
        const {reduxState, nonce, webvisor} = this.props;
        const {encryptedExperiments} = userSplitSelector(reduxState);
        const userInfo = getUserInfo(reduxState);

        /* global __DEV__ */

        if (__DEV__) {
            return null;
        }

        const metrikaParams: {
            id: number;
            clickmap: boolean;
            trackLinks: boolean;
            accurateTrackBounce: boolean;
            webvisor: boolean;
            ecommerce: boolean;
            nonce: string;
            encryptedExperiments: string;
            triggerEvent: boolean;
            defer?: boolean;
            trackHash?: boolean;
            type?: number;
        } = {
            id: METRIKA_COUNTER_ID,
            clickmap: true,
            trackLinks: true,
            accurateTrackBounce: true,
            webvisor,
            ecommerce: true,
            nonce,
            encryptedExperiments,
            triggerEvent: true,
        };

        const metrikaHtml = `
            <script type="text/javascript" nonce="${nonce}">
                (function(m,i){m[i]=m[i]||function(){(m[i].a=m[i].a||[]).push(arguments)};
                m[i].l=1*new Date()})(window, "ym");

                try {
                    ym(${metrikaParams.id}, "init", {
                        defer: ${Boolean(metrikaParams.defer)},
                        clickmap: ${Boolean(metrikaParams.clickmap)},
                        trackLinks: ${Boolean(metrikaParams.trackLinks)},
                        accurateTrackBounce: ${
                            metrikaParams.accurateTrackBounce
                        },
                        webvisor: ${Boolean(metrikaParams.webvisor)},
                        triggerEvent: ${Boolean(metrikaParams.triggerEvent)},
                        trackHash: ${Boolean(metrikaParams.trackHash)},
                        ecommerce: ${metrikaParams.ecommerce},
                        type: ${Number(metrikaParams.type)},
                        experiments: "${
                            metrikaParams.encryptedExperiments || ''
                        }"
                    });
                } catch(e) {}
            </script>

            <noscript>
                <div>
                    <img src="https://mc.yandex.ru/watch/${
                        metrikaParams.id
                    }" style="position:absolute; left:-9999px;" alt="" />
                </div>
            </noscript>

            <script type="text/javascript" nonce="${nonce}">
                document.addEventListener('yacounter${METRIKA_COUNTER_ID}inited', function() {
                    if (!window['yaCounter${METRIKA_COUNTER_ID}']) {
                        return;
                    }

                    if (typeof window['yaCounter${METRIKA_COUNTER_ID}Queue'] !== 'undefined') {
                        for (var i in window['yaCounter${METRIKA_COUNTER_ID}Queue']) {
                            if (window['yaCounter${METRIKA_COUNTER_ID}Queue'].hasOwnProperty(i)) {
                                var goalObject = window['yaCounter${METRIKA_COUNTER_ID}Queue'][i];

                                if (goalObject.target) {
                                    window['yaCounter${METRIKA_COUNTER_ID}'].reachGoal.apply(window['yaCounter${METRIKA_COUNTER_ID}'], [goalObject.target].concat(goalObject.props));
                                } else if (goalObject.user) {
                                    window['yaCounter${METRIKA_COUNTER_ID}'].userParams.call(window['yaCounter${METRIKA_COUNTER_ID}'], goalObject.props);
                                } else {
                                    window['yaCounter${METRIKA_COUNTER_ID}'].params.call(window['yaCounter${METRIKA_COUNTER_ID}'], goalObject.props);
                                }
                            }
                        }


                        delete(window['yaCounter${METRIKA_COUNTER_ID}Queue']);
                    }

                    const yandexUid = '${userInfo.yu}';

                    window['yaCounter${METRIKA_COUNTER_ID}'].setUserID(yandexUid);
                });
            </script>`;

        return <div dangerouslySetInnerHTML={{__html: metrikaHtml}} />;
    }

    private renderAppDataScript(): React.ReactNode {
        const {nonce, appData} = this.props;

        const appDataScript = `window.appData = ${serialize(appData)}`;
        const timeCorrectionScript = `window.timeCorrection = ${Date.now()} - Date.now()`;

        const clientScripts = [appDataScript, timeCorrectionScript];

        return (
            <script
                nonce={nonce}
                dangerouslySetInnerHTML={{__html: clientScripts.join(';')}}
            />
        );
    }

    private renderRumInlineScripts(): React.ReactNode {
        const {reduxState, nonce, appData, reqPath, reqQuery} = this.props;

        return (
            <Rum
                geoId={getUserGeoLocation(reduxState).geoId}
                nonce={nonce}
                environment={appData.appEnv}
                reqPath={reqPath}
                reqQuery={reqQuery}
                deviceType={reduxState.common.deviceType}
            />
        );
    }

    private renderYandexErrorsScripts(): React.ReactNode {
        const {
            appData,
            appVersion,
            nonce,
            reduxState: {
                common: {deviceType, experiments},
            },
        } = this.props;

        return (
            <YandexErrors
                appEnv={appData.appEnv}
                geoId={getUserGeoLocation(this.props.reduxState).geoId}
                nonce={nonce}
                deviceType={deviceType}
                experiments={experiments}
                appVersion={appVersion}
            />
        );
    }

    render(): React.ReactNode {
        const {
            reduxState,
            reduxState: {
                common: {deviceType},
            },
        } = this.props;

        const {[EToggler.FONTS_LOADED]: wasFontsLoadedBefore} =
            togglersSelector(reduxState);

        return (
            <html>
                <head>
                    {this.renderAppDataScript()}
                    {HtmlContainer.renderHelmetStaticAndMeta()}
                    {HtmlContainer.renderPreloadFonts(wasFontsLoadedBefore)}
                    {this.renderCriticalCss()}
                    {this.getPreloadScripts()}
                    {this.renderRumInlineScripts()}
                    {this.renderYandexErrorsScripts()}
                </head>
                <body
                    className={classnames({
                        deviceTypeIOS: isOS(deviceType.os, IOS),
                        mobile: deviceType.isMobile,
                        fontsLoaded: wasFontsLoadedBefore,
                    })}
                >
                    {this.getRootNode()}
                    {this.renderMetrikaCounter()}
                    {this.renderScriptWithReduxState()}
                    {this.getChunksScripts()}
                </body>
            </html>
        );
    }
}
