import {
    ICurrenciesInfo,
    ICurrencyServerInfo,
} from 'server/services/CurrenciesService/types';

import {DEFAULT_CURRENCIES} from 'reducers/common/currencies/reducer';

import {CurrencyType} from 'utilities/currency/CurrencyType';
import {IAviaBackendApiClient} from 'server/utilities/AviaBackendApiClient/IAviaBackendApiClient';

import {IDependencies, INationalSettings} from 'server/getContainerConfig';

import {IGeobaseService} from '../GeobaseService/GeobaseService';

export class CurrenciesService {
    private readonly aviaGatewayBackendApiClient: IAviaBackendApiClient;
    private readonly preferredCurrency: CurrencyType | undefined;
    private readonly nationalSettings: INationalSettings;
    private readonly geobaseService: IGeobaseService;

    constructor({
        preferredCurrency,
        aviaGatewayBackendApiClient,
        nationalSettings,
        geobaseService,
    }: IDependencies) {
        this.preferredCurrency = preferredCurrency;
        this.aviaGatewayBackendApiClient = aviaGatewayBackendApiClient;
        this.nationalSettings = nationalSettings;
        this.geobaseService = geobaseService;
    }

    async getCurrencyInfoByUserGeoLocation(): Promise<ICurrenciesInfo> {
        const {geoId} = await this.geobaseService.getUserGeoLocation();

        if (!geoId) {
            return DEFAULT_CURRENCIES;
        }

        const currencies = await this.fetchCurrencies(geoId);

        if (!currencies) {
            return DEFAULT_CURRENCIES;
        }

        return currencies;
    }

    async getCurrencyInfo(geoId: number): Promise<ICurrenciesInfo> {
        const currencies = await this.fetchCurrencies(geoId);

        if (!currencies) {
            return DEFAULT_CURRENCIES;
        }

        return currencies;
    }

    private async fetchCurrencies(
        geoId: number,
    ): Promise<ICurrenciesInfo | null> {
        // TODO: по результатам стрельб, возможно, понадобится добавить кэш
        const rawCurrencies =
            await this.aviaGatewayBackendApiClient.request<ICurrencyServerInfo>(
                {
                    name: 'currency',
                    params: {
                        geoId,
                    },
                },
            );

        if (!rawCurrencies) {
            return null;
        }

        const {rates} = rawCurrencies;
        const {nationalCurrency, allowedCurrencies} = this.nationalSettings;
        const preferredCurrency = this.getPreferredCurrency();

        return {
            nationalCurrency: nationalCurrency,
            preferredCurrency: rates[preferredCurrency]
                ? preferredCurrency
                : nationalCurrency,
            availableCurrencies: allowedCurrencies,
            currencyRates: rates,
        };
    }

    private getPreferredCurrency(): CurrencyType {
        const {allowedCurrencies, nationalCurrency} = this.nationalSettings;

        if (
            this.preferredCurrency &&
            allowedCurrencies.includes(this.preferredCurrency)
        ) {
            return this.preferredCurrency;
        }

        return nationalCurrency;
    }
}
