import {TGeoId} from 'types/TGeoId';

import {IGeoLocation} from 'reducers/common/commonReducerTypes';

import {ILogger} from 'server/utilities/Logger';

import {IDependencies} from 'server/getContainerConfig';
import {GeobaseApi} from 'server/api/GeobaseApi/GeobaseApi';

export interface IGeobaseService {
    getUserGeoLocation: (
        forcedRegionId?: string,
    ) => Promise<Required<IGeoLocation>>;
}

const DEFAULT_GEOLOCATION: Required<IGeoLocation> = {
    geoName: 'Москва',
    geoId: 213 as TGeoId,
    countryId: 225,
};

export class GeobaseService implements IGeobaseService {
    private geobaseApi: GeobaseApi;
    private logger: ILogger;

    constructor({geobaseApi, logger}: IDependencies) {
        this.geobaseApi = geobaseApi;
        this.logger = logger;
    }

    async getUserGeoLocation(
        forcedRegionId?: string,
    ): Promise<Required<IGeoLocation>> {
        const regionId = await this.getRegionId(forcedRegionId);

        if (!regionId) {
            return this.handleGeolocationError();
        }

        const [region, countryId] = await Promise.all([
            this.geobaseApi.getRegionById(regionId),
            this.geobaseApi.findCountry(regionId),
        ]);

        if (!region || !countryId) {
            return this.handleGeolocationError();
        }

        return {
            geoName: region.name,
            geoId: region.id,
            countryId,
        };
    }

    async getRegionId(forcedRegionId?: string): Promise<number> {
        if (forcedRegionId) {
            return parseInt(forcedRegionId, 10);
        }

        const geolocation = await this.geobaseApi.getGeoLocation();

        if (!geolocation) {
            return this.handleGeolocationError().geoId;
        }

        // eslint-disable-next-line camelcase
        return geolocation.region_id;
    }

    private handleGeolocationError(): Required<IGeoLocation> {
        this.logger.logWarn('geobase: downgrade to default geolocation');

        return DEFAULT_GEOLOCATION;
    }
}
