import cuid from 'cuid';

import {EApiEntry} from 'types/EApiEntry';

import {IDependencies} from 'server/getContainerConfig';

import {RestApiClient} from '../RestApiClient';
import {
    IAviaBackendApiClient,
    IAviaBackendRequest,
} from './IAviaBackendApiClient';
import {IRequestConfig} from '../HttpClient/IHttpClient';
import AviaBackendApiError from './AviaBackendApiError';

const DEFAULT_TIMEOUT = 5000;

interface IResponse<T> {
    [key: string]: any;
    status: string;
    data: T;
}

export class AviaBackendRestApiClient
    extends RestApiClient
    implements IAviaBackendApiClient
{
    private yandexuid: string;
    private passportid: string;
    private readonly nationalVersion: string;

    constructor({
        httpClient,
        getApiHost,
        getSrcParams,
        logger,
        blackbox,
        yandexuid,
        requestId,
        nationalVersion,
        rootSpan,
    }: IDependencies) {
        super({
            baseURL: getApiHost(EApiEntry.AVIA_BACKEND),
            httpClient,
            logger,
            requestId,
            rootSpan,
        });

        this.srcParams = getSrcParams(EApiEntry.AVIA_BACKEND);
        this.yandexuid = yandexuid;
        this.passportid = blackbox.uid;
        this.nationalVersion = nationalVersion;
    }

    async request<T>(
        aviaBackendRequest: IAviaBackendRequest,
    ): Promise<T | null> {
        const params = this.extractQueryParams(aviaBackendRequest);

        return this.sendAndValidateRequest<T>(
            {
                params,
                headers: {
                    'user-agent': 'travel-frontend',
                    'content-type': 'application/json',
                },
                timeout: DEFAULT_TIMEOUT,
            },
            aviaBackendRequest,
        ).catch(ex => {
            /** ignore error */
            this.logger.logError('AviaBackendRestApiClient failure', ex);

            return null;
        });
    }

    private async sendAndValidateRequest<T>(
        options: IRequestConfig,
        aviaBackendRequest: IAviaBackendRequest,
    ): Promise<T> {
        const path = aviaBackendRequest.name;
        let json;

        try {
            json = await this.get<IResponse<T>>(`/rest/${path}`, options);
        } catch (err) {
            throw new AviaBackendApiError(err, aviaBackendRequest, '');
        }

        // Даже с кодом 200 может быть ошибка
        if (!json || !json.status || json.status !== 'ok') {
            throw new AviaBackendApiError(
                'Внутренняя ошибка API',
                aviaBackendRequest,
                JSON.stringify(json),
            );
        }

        return json.data;
    }

    private extractQueryParams(aviaBackend: IAviaBackendRequest) {
        return {
            national_version: this.nationalVersion,
            lang: 'ru',

            // ID запроса нужен для логирования
            cuid: cuid(),

            // Инфа о пользователя нужна для логирования
            yandexuid: this.yandexuid,
            passportuid: this.passportid,

            name: aviaBackend.name,
            ...aviaBackend.params,
        };
    }
}
