import {ParsedQuery} from 'query-string';
import {pick, mapValues, flatMap, isEqual, identity, pickBy} from 'lodash';

import {
    monthlyAttributionFields,
    sessionAttributionFields,
} from 'server/utilities/DataStorage/AttributionData/constants/attributionFields';
import mapFieldToQuery from 'server/utilities/DataStorage/AttributionData/constants/mapFieldToQuery';

import {
    ECookieName,
    TCookiesMeta,
} from 'server/utilities/DataStorage/AttributionData/types/ICookie';
import IAttributionDataValues from 'server/utilities/DataStorage/AttributionData/types/IAttributionDataValues/IAttributionDataValues';
import IAttributionTrainsDataValues from 'server/utilities/DataStorage/AttributionData/types/IAttributionDataValues/IAttributionTrainsDataValues';
import IAttributionUTMDataValues from 'server/utilities/DataStorage/AttributionData/types/IAttributionDataValues/IAttributionUTMDataValues';
import IAttributionData from 'server/utilities/DataStorage/AttributionData/types/IAttributionData';
import {ICookies, Response} from '@yandex-data-ui/core/lib/types';
import {TNormalizedQuery} from 'server/utilities/DataStorage/AttributionData/types/TNormalizedQuery';
import IAttributionUaas from 'server/utilities/DataStorage/AttributionData/types/IAttributionUaas';

import getOrganicSource from 'server/utilities/utm/utils/getOrganicSource';
import {ILogger} from 'server/utilities/Logger';
import {getNow} from 'utilities/dateUtils';

import {IDependencies} from 'server/getContainerConfig';

type TAttributionDataParams = Pick<
    IDependencies,
    'logger' | 'query' | 'cookies' | 'setCookie' | 'referrer' | 'testBuckets'
>;

/**
 * Хранилище липких данных о пользователе
 */
export default class AttributionData implements IAttributionData {
    static readonly COOKIES_META: TCookiesMeta = {
        /**
         * session
         */
        [ECookieName.TRAVEL_ATTRIBUTION_SESSION]: {
            expires: 0,
            fields: sessionAttributionFields,
        },
        /**
         * 30 days
         */
        [ECookieName.TRAVEL_ATTRIBUTION_MONTHLY]: {
            expires: 1000 * 60 * 60 * 24 * 30,
            fields: monthlyAttributionFields,
        },
    };

    private readonly query: ParsedQuery = {};
    private readonly cookies: ICookies = {};
    private readonly setCookie: Response['cookie'];
    private readonly logger: ILogger;
    private readonly referrer: string | undefined;
    private readonly testBuckets: string | undefined;

    private data: IAttributionDataValues = {};

    constructor({
        logger,
        query,
        cookies,
        setCookie,
        referrer,
        testBuckets,
    }: TAttributionDataParams) {
        this.logger = logger;
        this.query = query;
        this.cookies = cookies;
        this.setCookie = setCookie;
        this.referrer = referrer;
        this.testBuckets = testBuckets;
    }

    getUtm(): IAttributionUTMDataValues {
        const {utmSource, utmMedium, utmCampaign, utmTerm, utmContent} =
            this.data;

        return {
            utmSource,
            utmMedium,
            utmCampaign,
            utmTerm,
            utmContent,
        };
    }

    getSerpReqId(): string | undefined {
        return this.data.serpReqId;
    }

    getSerpUuid(): string | undefined {
        return this.data.serpUuid;
    }

    getSerpTestId(): string | undefined {
        return this.data.serpTestId;
    }

    getGclid(): string | undefined {
        return this.data.gclid;
    }

    getYclid(): string | undefined {
        return this.data.yclid;
    }

    getFbclid(): string | undefined {
        return this.data.fbclid;
    }

    getYtpReferer(): string | undefined {
        return this.data.ytpReferer;
    }

    getFrom(): string | undefined {
        return this.data.from;
    }

    getUaas(): IAttributionUaas {
        const {testBuckets} = this;

        return {
            testBuckets,
        };
    }

    getWizardFlags(): string | undefined {
        return this.data.wizardFlags;
    }

    getWizardredirkey(): string | undefined {
        return this.data.wizardRedirKey;
    }

    getPartner(): IAttributionTrainsDataValues {
        const {partner, subpartner, partnerUid} = this.data;

        return {
            partner,
            subpartner,
            partnerUid,
        };
    }

    getUserRegion(): string | undefined {
        return this.data.userRegion;
    }

    updateCookiesAndFillData(): void {
        const referrerDataValues = this.loadDataValuesFromReferrer();
        const cookiesDataValues = this.loadDataValuesFromCookies();
        const queryDataValues = this.loadDataValuesFromQuery();

        const dataValues = {
            ...pickBy(referrerDataValues, identity),
            ...pickBy(cookiesDataValues, identity),
            ...pickBy(queryDataValues, identity),
        };

        Object.values(ECookieName).forEach(cookieName => {
            this.saveCookie(cookieName, dataValues);
        });

        this.data = dataValues;
    }

    private loadDataValuesFromCookies(): IAttributionDataValues {
        return Object.values(ECookieName).reduce((acc, cookieName) => {
            return {
                ...acc,
                ...this.parseCookie(cookieName),
            };
        }, {} as IAttributionDataValues);
    }

    private loadDataValuesFromReferrer(): IAttributionDataValues {
        const referrer = this.referrer;
        const utmSource = getOrganicSource(referrer);

        return {
            utmSource,
        };
    }

    private loadDataValuesFromQuery(): IAttributionDataValues {
        const fields = flatMap(
            Object.values(AttributionData.COOKIES_META).map(
                meta => meta.fields,
            ),
        );

        return fields.reduce<IAttributionDataValues>((acc, field) => {
            const queryParamNames = mapFieldToQuery[field];

            acc[field] = this.getFirstExistingValueFromQuery(
                this.query,
                queryParamNames,
            );

            return acc;
        }, {});
    }

    private getFirstExistingValueFromQuery(
        query: ParsedQuery,
        queryParamNames: string[],
    ): string | undefined {
        return this.normalizeValue(
            queryParamNames.map(name => query[name]).find(identity),
        );
    }

    /**
     * Сохраняет в куках зафиксированные в COOKIES_META поля с зафиксированным expires
     */
    private saveCookie(
        cookieName: ECookieName,
        dataValues: IAttributionDataValues,
    ): void {
        const meta = AttributionData.COOKIES_META[cookieName];
        const {expires, fields} = meta;

        const normalizedDataValues = this.normalize(dataValues);
        const suitableDataValues = pick(normalizedDataValues, fields);

        const currentCookieValue = this.parseCookie(cookieName);

        if (isEqual(suitableDataValues, currentCookieValue)) {
            return;
        }

        this.setCookie(cookieName, JSON.stringify(suitableDataValues), {
            expires: expires > 0 ? new Date(getNow() + expires) : undefined,
        });
    }

    private parseCookie(cookieName: ECookieName): IAttributionDataValues {
        try {
            const cookieValue = this.cookies[cookieName];

            if (!cookieValue) {
                return {};
            }

            return JSON.parse(cookieValue);
        } catch (e) {
            this.logger.logWarn(
                `Failed load AttributionData from cookie: ${cookieName}`,
            );

            return {};
        }
    }

    /**
     * убираем массивы и null из параметров
     * https://st.yandex-team.ru/TRAVELFRONT-3334
     */
    private normalize(query: IAttributionDataValues): TNormalizedQuery {
        return mapValues(query, val => this.normalizeValue(val));
    }

    private normalizeValue(
        value: string | string[] | null | undefined,
    ): string | undefined {
        if (!value) {
            return undefined;
        }

        if (Array.isArray(value)) {
            this.logger.logWarn('Utm label contains array');

            return value[0];
        }

        return value;
    }
}
