import axios, {AxiosInstance} from 'axios';
import axiosRetry from 'axios-retry';
import https from 'https';

import {CommonHeaders} from 'server/constants/headers';

import {TRawExperiments} from 'server/providers/experiments/types';
import IUserDataValues from 'server/utilities/DataStorage/UserData/types/IUserDataValues';

import {IDependencies} from 'server/getContainerConfig';

import {IHttpClient, IRequestConfig, IResponse} from './IHttpClient';

const {
    X_REQUEST_ID,
    X_YA_LOGIN,
    X_YA_PASSPORTID,
    X_YA_SESSION_KEY,
    X_YA_USER_TICKET,
    X_YA_YANDEXUID,
    X_YA_UAAS_EXPERIMENTS,
    X_YA_USER_IP,
    X_YA_EXP_BOXES,
    X_YA_USER_DEVICE,
    X_YA_ICOOKIE,
    X_YA_USER_GID,
    X_YA_USER_AGENT,
    X_YA_USER_IS_PLUS,
    X_YA_USER_IS_STAFF,
} = CommonHeaders;

export class HttpClient implements IHttpClient {
    private readonly sessionKey: string;

    private readonly requestId: string;

    private readonly requestIp: string;

    private readonly yandexuid?: string;

    private readonly uid?: string;

    private readonly userTicket?: string;

    private readonly login?: string;

    private readonly rawExperiments?: TRawExperiments;

    private readonly axiosInstance: AxiosInstance;

    private readonly testBuckets?: string;

    private readonly deviceType?: IUserDataValues['deviceType'];

    private readonly iCookie?: IUserDataValues['iCookie'];

    private readonly yandexGid?: string;

    private readonly userAgent?: string;

    private readonly isStaff?: boolean;

    private readonly havePlus?: boolean;

    constructor({
        sessionKey,
        requestId,
        requestIp,
        userInfo,
        userAgent,
        yandexuid,
        yandexGid,
        blackbox,
        attributionData,
        userData,
        rawExperiments,
    }: IDependencies) {
        this.sessionKey = sessionKey;
        this.requestId = requestId;
        this.requestIp = requestIp;
        this.yandexuid = yandexuid;
        this.yandexGid = yandexGid;
        this.userAgent = userAgent;
        this.rawExperiments = rawExperiments;
        this.testBuckets = attributionData.getUaas().testBuckets;
        this.iCookie = userData.getICookie();
        this.deviceType = userData.getDeviceType();

        this.axiosInstance = axios.create({
            httpsAgent: new https.Agent({
                rejectUnauthorized: process.env.NODE_ENV === 'production',
            }),
        });

        if (userInfo && 'uid' in userInfo) {
            this.uid = userInfo.uid;
            this.login = userInfo.login || undefined;
            this.userTicket = blackbox.userTicket;
            this.isStaff = userInfo.isStaff;
            this.havePlus = userInfo.havePlus;
        }
    }

    request<T = any>(config: IRequestConfig): Promise<IResponse<T>> {
        this.patchConfigWithCommonHeaders(config);

        axiosRetry(this.axiosInstance, config['axios-retry'] || {retries: 0});

        return this.axiosInstance.request(config);
    }

    private patchConfigWithCommonHeaders(config: IRequestConfig): void {
        const headers = {...config.headers};

        if (!headers[X_YA_SESSION_KEY] && !this.uid) {
            headers[X_YA_SESSION_KEY] = this.sessionKey;
        }

        if (!headers[X_REQUEST_ID]) {
            headers[X_REQUEST_ID] = this.requestId;
        }

        if (!headers[X_YA_USER_TICKET] && this.userTicket) {
            headers[X_YA_USER_TICKET] = this.userTicket;
        }

        if (!headers[X_YA_YANDEXUID] && this.yandexuid) {
            headers[X_YA_YANDEXUID] = this.yandexuid;
        }

        if (!headers[X_YA_PASSPORTID] && this.uid) {
            headers[X_YA_PASSPORTID] = this.uid;
        }

        if (!headers[X_YA_LOGIN] && this.login) {
            headers[X_YA_LOGIN] = this.login;
        }

        if (!headers[X_YA_UAAS_EXPERIMENTS] && this.rawExperiments) {
            headers[X_YA_UAAS_EXPERIMENTS] = JSON.stringify(
                this.rawExperiments,
            );
        }

        if (!headers[X_YA_USER_IP] && this.requestIp) {
            headers[X_YA_USER_IP] = this.requestIp;
        }

        if (!headers[X_YA_EXP_BOXES] && this.testBuckets) {
            headers[X_YA_EXP_BOXES] = this.testBuckets;
        }

        if (!headers[X_YA_USER_DEVICE]) {
            headers[X_YA_USER_DEVICE] = this.deviceType;
        }

        if (!headers[X_YA_ICOOKIE] && this.iCookie) {
            headers[X_YA_ICOOKIE] = this.iCookie;
        }

        if (!headers[X_YA_USER_GID] && this.yandexGid) {
            headers[X_YA_USER_GID] = this.yandexGid;
        }

        if (!headers[X_YA_USER_AGENT] && this.userAgent) {
            headers[X_YA_USER_AGENT] = this.userAgent;
        }

        if (!headers[X_YA_USER_IS_PLUS] && this.havePlus) {
            headers[X_YA_USER_IS_PLUS] = this.havePlus;
        }

        if (!headers[X_YA_USER_IS_STAFF] && this.isStaff) {
            headers[X_YA_USER_IS_STAFF] = this.isStaff;
        }

        config.headers = headers;
    }
}
