/**
 * @typedef {import('@yandex-data-ui/core/lib/types').Request} Request
 * @typedef {import('@yandex-data-ui/core/lib/types').Response} Response
 * @typedef {import('@yandex-data-ui/core/lib/types').NextFunction} NextFunction
 * @typedef {import('./types/IUtmMap').IUtmMap} IUtmMap
 */

const MARKS = require('./constants/marks');

const parseUtm = require('./utils/parseUtm');
const validateClid = require('./utils/validateClid');
const getOrganicSource = require('./utils/getOrganicSource');
const getCookieParams = require('./utils/getCookieParams');
const toggleCookie = require('./utils/toggleCookie');
const prepareValue = require('./utils/prepareValue');
const shouldFinishUtmSession = require('./utils/shouldFinishUtmSession');
const logInvalidQueryTypes = require('./utils/logInvalidQueryTypes');
const {
    getWizardFlagsFromQuery,
} = require('../../../src/utilities/url/wizard/getWizardFlagsFromQuery');
const {
    getWizardReqIdFromQuery,
} = require('../../../src/utilities/url/wizard/getWizardReqIdFromQuery');

/**
 * Комбинирует метки из параметров запроса и кук, используется в мидлваре.
 *
 * @param {Request} req
 * @param {Response} res
 * @returns {undefined}
 */
function register(req, res) {
    logInvalidQueryTypes(req);

    const {query, cookies} = req;

    const queryUtms = parseUtm(req.query);

    let hasQueryUtms = Object.keys(queryUtms).length > 0;

    const queryClid = query[MARKS.CLID],
        queryStid = query[MARKS.STID],
        queryWizardFlags = getWizardFlagsFromQuery(query),
        querySerpUuid = query[MARKS.SERP_UUID],
        queryWizardReqId = getWizardReqIdFromQuery(query);

    let cookieWizardReqId24 = cookies[MARKS.WIZARD_REQ_ID24],
        cookieWizardFlags = cookies[MARKS.WIZARD_FLAGS],
        cookieWizardReqId = cookies[MARKS.WIZARD_REQ_ID],
        cookieSerpUuid = cookies[MARKS.SERP_UUID],
        cookiesUtmSession = cookies[MARKS.UTM_SESSION],
        utms = cookies[MARKS.UTMS],
        cookieClid = cookies[MARKS.CLID],
        cookieStid = cookies[MARKS.STID],
        utm24h = Boolean(Number(cookies[MARKS.UTMS_24H]));

    const validClid = validateClid(queryClid);

    /**
     * В cookies.utms лежит IAviaUtmMap в json.
     * Если значение сломали, то затираем эту куку.
     */
    let cookieUtms = {};

    try {
        if (utms) {
            cookieUtms = JSON.parse(utms);
        }
    } catch (e) {
        res.aviaClearCookie(MARKS.UTMS);
    }

    const currentUtms = hasQueryUtms ? queryUtms : cookieUtms;

    /**
     * Когда пользователь пришел без меток, мапим реферрер на фиктивные метки
     * и дальше обрабатываем так, будто метки есть (если реферрер замапился)
     */
    if (Object.keys(currentUtms).length === 0) {
        const source = getOrganicSource(req.get('referrer'));

        if (source) {
            hasQueryUtms = true;
            queryUtms.utm_source = source;
        }
    }

    if (hasQueryUtms) {
        // utm метки складываем на 10 дней
        res.aviaCookie(
            MARKS.UTMS,
            JSON.stringify(queryUtms),
            getCookieParams(10),
        );

        // дополнительно ставим флаг для меток на 24 часа
        res.aviaCookie(MARKS.UTMS_24H, '1', getCookieParams(1));
        utm24h = true;

        toggleCookie(res, query, cookies, MARKS.CLID, validClid);
        toggleCookie(res, query, cookies, MARKS.STID, queryStid);

        // если в куках есть utmSession, то нужно сбросить ее, т.к. она появляется после целевого действия,
        // а здесь мы получли новые метки и utmSession будет неактуалной
        if (cookiesUtmSession) {
            res.aviaClearCookie(MARKS.UTM_SESSION);
            delete cookies[MARKS.UTM_SESSION];
        }

        // ютм метки из запроса приоритетней метки колдуна в куках
        if (cookieWizardReqId24) {
            res.aviaClearCookie(MARKS.WIZARD_REQ_ID24);
            cookieWizardReqId24 = null;
        }

        cookieClid = cookieStid = cookiesUtmSession = null;
    }

    if (shouldFinishUtmSession(cookiesUtmSession)) {
        if (cookieUtms) {
            res.aviaClearCookie(MARKS.UTMS);
        }

        if (cookiesUtmSession) {
            res.aviaClearCookie(MARKS.UTM_SESSION);
        }

        cookieClid = cookieStid = cookiesUtmSession = null;
    }

    if (queryWizardFlags) {
        res.aviaCookie(
            MARKS.WIZARD_FLAGS,
            queryWizardFlags,
            getCookieParams(1),
        );
        cookieWizardFlags = queryWizardFlags;
    }

    if (queryWizardReqId) {
        res.aviaCookie(MARKS.WIZARD_REQ_ID24, '1', getCookieParams(1));
        cookieWizardReqId24 = '1';

        res.aviaCookie(
            MARKS.WIZARD_REQ_ID,
            queryWizardReqId,
            getCookieParams(1),
        );
        cookieWizardReqId = queryWizardReqId;
    }

    if (querySerpUuid) {
        res.aviaCookie(MARKS.SERP_UUID, querySerpUuid, getCookieParams(1));
        cookieSerpUuid = querySerpUuid;
    }

    utms = parseUtm(currentUtms);

    const clidValue = validClid || cookieClid;
    const stidValue = query.stid || cookieStid;

    req.visitSourceIds = {
        [MARKS.CLID]: prepareValue(clidValue),
        [MARKS.STID]: prepareValue(stidValue),

        [MARKS.WIZARD_FLAGS]: prepareValue(cookieWizardFlags),
        [MARKS.WIZARD_REQ_ID]: prepareValue(cookieWizardReqId),
        [MARKS.WIZARD_REQ_ID24]: Boolean(cookieWizardReqId24),
        [MARKS.SERP_UUID]: prepareValue(cookieSerpUuid),

        [MARKS.UTMS_24H]: utm24h ? {...utms} : null,
        [MARKS.CLID_24H]: prepareValue(utm24h ? clidValue : null),
        [MARKS.STID_24H]: prepareValue(utm24h ? stidValue : null),
    };

    req[MARKS.UTMS] = utms;
}

/**
 * Чистит метки, используется в целевом действии.
 *
 * @param {Request} req
 * @param {Response} res
 * @returns {undefined}
 */
function track(req, res) {
    const clearCookie = mark => {
        if (req.cookies[mark] || res.hasCookie(mark)) {
            res.aviaClearCookie(mark);
        }
    };

    if (!req.cookies[MARKS.UTM_SESSION] && req.cookies[MARKS.UTMS]) {
        // Метки, для которых нужно сбросить expire (aka жизнь до закрытия браузера)
        const cookiesToUpdate = [MARKS.UTMS, MARKS.CLID, MARKS.STID];

        const params = {
            httpOnly: true,
        };

        cookiesToUpdate
            .filter(name => Boolean(req.cookies[name]))
            .forEach(name => res.aviaCookie(name, req.cookies[name], params));

        // Эта кука хранит время, которое нужно хранить метки в cookies.utms. Это время проверяется в
        // функции track
        res.aviaCookie(MARKS.UTM_SESSION, Date.now(), params);
    }

    const marksToClear = [
        MARKS.UTMS_24H,
        MARKS.WIZARD_FLAGS,
        MARKS.WIZARD_REQ_ID,
        MARKS.SERP_UUID,
    ];

    marksToClear.forEach(clearCookie);
}

module.exports = {
    register,
    track,
};
