import React, {ReactNode} from 'react';

import {IWithClassName} from 'types/withClassName';
import {IIconProps} from 'icons/types/icon';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {deviceMods} from 'utilities/stylesUtils';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import {CardWithDeviceLayout} from 'components/CardWithDeviceLayout/CardWithDeviceLayout';
import Swipeable, {
    ISwipeableRenderItemParams,
} from 'components/Swipeable/Swipeable';
import Advantage from 'components/Advantages/components/Advantage/Advantage';

import cx from './Advantages.scss';

type TAdvantage = {
    title: string;
    description: React.ReactNode;
    IconComponent: React.FC<IIconProps>;
    iconWidth: number;
    iconHeight: number;
};

interface IAdvantagesProps extends IWithClassName, IWithQaAttributes {
    advantages: TAdvantage[];
}

const Advantages: React.FC<IAdvantagesProps> = props => {
    const {className, advantages} = props;

    const deviceType = useDeviceType();

    const renderer = ({
        data,
        index,
    }: Omit<
        ISwipeableRenderItemParams<TAdvantage>,
        'isCurrent'
    >): ReactNode => (
        <Advantage
            key={index}
            className={cx('item')}
            title={data.title}
            description={data.description}
            IconComponent={data.IconComponent}
            iconWidth={data.iconWidth}
            iconHeight={data.iconHeight}
            {...prepareQaAttributes({
                parent: props,
                current: 'advantage',
                key: index,
            })}
        />
    );

    const renderContent = () => {
        if (deviceType.isMobile) {
            return (
                <Swipeable
                    className={cx('advantagesSwipeable')}
                    sideThreshold={0.1}
                    renderItem={renderer}
                    itemsData={advantages}
                    showNav
                />
            );
        }

        return (
            <>
                {advantages.map((data: TAdvantage, index: number) =>
                    renderer({data, index}),
                )}
            </>
        );
    };

    return (
        <CardWithDeviceLayout
            className={cx('root', deviceMods('root', deviceType), className)}
        >
            {renderContent()}
        </CardWithDeviceLayout>
    );
};

export default React.memo(Advantages);
