import React, {useMemo} from 'react';

import {IWithClassName} from 'types/withClassName';

import {IWithQaAttributes} from 'utilities/qaAttributes/qaAttributes';

import ExcludeIcon from 'icons/16/Exclude';
import Flex from 'components/Flex/Flex';
import Text from 'components/Text/Text';
import BaggageIcon from 'components/BaggageIconWithWeight/components/BaggageIcon/BaggageIcon';
import CarryOnIcon from 'components/BaggageIconWithWeight/components/CarryOnIcon/CarryOnIcon';

import cx from './BaggageIconWithWeight.scss';

export enum EBaggageWithWeightIconType {
    BAGGAGE = 'baggage',
    CARRY_ON = 'carryOn',
}

export enum EBaggageWithWeightIconSize {
    S = 's',
    M = 'm',
}

interface IBaggageIconWithWeightProps
    extends IWithClassName,
        IWithQaAttributes {
    type: EBaggageWithWeightIconType;
    included: boolean;
    weight?: number;
    count?: number;
    iconSize?: EBaggageWithWeightIconSize;
    allowRuble?: boolean;
    currencyPrefix?: string;
    alignItems?: React.CSSProperties['alignItems'];
}

const BaggageIconWithWeight: React.FC<IBaggageIconWithWeightProps> = props => {
    const {
        className,
        type,
        weight,
        included,
        count,
        iconSize = EBaggageWithWeightIconSize.M,
        allowRuble = false,
        currencyPrefix = '',
        alignItems = 'center',
        ...rest
    } = props;

    const content = useMemo(() => {
        if (!included || weight) {
            const size = iconSize === EBaggageWithWeightIconSize.M ? 11 : 8;

            return (
                <div className={cx('content')}>
                    {weight ?? <ExcludeIcon width={size} height={size} />}
                </div>
            );
        }

        return null;
    }, [iconSize, included, weight]);

    const icon = useMemo(
        () =>
            React.createElement(
                type === EBaggageWithWeightIconType.BAGGAGE
                    ? BaggageIcon
                    : CarryOnIcon,
                {
                    size: iconSize,
                    empty: Boolean(!included || weight),
                },
            ),
        [iconSize, included, type, weight],
    );

    return (
        <Flex
            className={cx(
                'root',
                `root_size_${iconSize}`,
                included ? 'root_included' : 'root_notIncluded',
                className,
            )}
            alignItems={alignItems}
            inline
            nowrap
            {...rest}
        >
            <div className={cx('iconWrapper')}>
                {icon}
                {content}
            </div>

            {count && count > 1 ? (
                <Text className={cx('count')} size="m" color="current">
                    x{count}
                </Text>
            ) : null}
            {!included && allowRuble && (
                <Text className={cx('rubble')} size="m">
                    {`${currencyPrefix}₽`}
                </Text>
            )}
        </Flex>
    );
};

export default React.memo(BaggageIconWithWeight);
