import {FunctionComponent, ReactNode} from 'react';
import {FieldMetaState} from 'react-final-form';

import {EFieldName} from './constants/fieldNames';

import {IWithClassName} from 'types/withClassName';
import {TValidator, IFieldError} from 'types/common/validation/form';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import * as i18nBlock from 'i18nNew/components-BookingContactsForm';

import Input from 'components/Input/Input';
import Heading from 'components/Heading/Heading';
import FormField from 'components/FormField/FormField';

import cx from './BookingContactsForm.scss';

export type TBookingContacts = Partial<Record<EFieldName, string>>;

export type TBookingContactErrors = Partial<Record<EFieldName, IFieldError>>;

export interface IFieldsValidation<TAllValues extends object = object> {
    [EFieldName.email]?: TValidator<string, TAllValues>;
    [EFieldName.phone]?: TValidator<string, TAllValues>;
}

export interface IBookingContactsFormProps
    extends IWithQaAttributes,
        IWithClassName {
    description?: string;
    disabled?: boolean;
    fieldClassName?: string;
    fieldsValidation?: IFieldsValidation;
    initialValues?: TBookingContacts;
    onChangeError?: (fieldName: string, meta: FieldMetaState<string>) => void;
}

const BookingContactsForm: FunctionComponent<IBookingContactsFormProps> = ({
    className,
    description,
    disabled,
    fieldsValidation,
    initialValues,
    fieldClassName,
    onChangeError,
    ...rest
}) => {
    const deviceType = useDeviceType();
    const {isMobile} = deviceType;

    return (
        <div className={cx({root_mobile: isMobile})}>
            <Heading className={cx('title', className)} level="3">
                {i18nBlock.title()}
            </Heading>
            <div className={cx('description')}>
                {description || i18nBlock.description()}
            </div>
            <div className={cx('fields')} {...prepareQaAttributes(rest)}>
                <FormField
                    className={cx('emailInput', fieldClassName)}
                    name={EFieldName.email}
                    title={i18nBlock.emailDashTitle()}
                    deviceType={deviceType}
                    initialValue={initialValues?.email}
                    validate={fieldsValidation?.email}
                    onChangeError={onChangeError}
                    control={(
                        {input},
                        {error, inputRef, controlRef},
                    ): ReactNode => (
                        <Input
                            {...input}
                            id={input.name}
                            inputRef={(currentInputRef): void => {
                                inputRef.current = currentInputRef;
                            }}
                            innerRef={controlRef}
                            value={input.value || null}
                            size="l"
                            state={error ? 'error' : undefined}
                            type="text"
                            inputMode="email"
                            disabled={disabled}
                            disabledAutocompleteFix
                            {...prepareQaAttributes({
                                parent: rest,
                                current: 'email',
                            })}
                        />
                    )}
                />

                <FormField
                    className={cx('phoneInput', fieldClassName)}
                    name={EFieldName.phone}
                    title={i18nBlock.phoneDashTitle()}
                    deviceType={deviceType}
                    initialValue={initialValues?.phone}
                    validate={fieldsValidation?.phone}
                    onChangeError={onChangeError}
                    control={(
                        {input},
                        {error, inputRef, controlRef},
                    ): ReactNode => (
                        <Input
                            {...input}
                            id={input.name}
                            inputRef={(currentInputRef): void => {
                                inputRef.current = currentInputRef;
                            }}
                            innerRef={controlRef}
                            value={input.value || null}
                            size="l"
                            type="text"
                            inputMode="tel"
                            state={error ? 'error' : undefined}
                            disabled={disabled}
                            disabledAutocompleteFix
                            {...prepareQaAttributes({
                                parent: rest,
                                current: 'phone',
                            })}
                        />
                    )}
                />
            </div>
        </div>
    );
};

export default BookingContactsForm;
