import React, {Dispatch, SetStateAction} from 'react';

import {IWithClassName} from 'types/withClassName';

import {
    prepareQaAttributes,
    IWithQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import {
    TextColor,
    TTextWeight,
    TTextSize,
    textColorCn,
    textSizeCn,
    textWeightCn,
} from '../Text/Text';

import cx from './Box.scss';

export type TBoxSizes =
    | '0'
    | '1'
    | '2'
    | '3'
    | '4'
    | '5'
    | '6'
    | '7'
    | '8'
    | '9'
    | '10'
    | 0
    | 1
    | 2
    | 3
    | 4
    | 5
    | 6
    | 7
    | 8
    | 9
    | 10;

export interface IBoxProps extends IWithClassName, IWithQaAttributes {
    /**
     * Тег ноды
     *
     * @default 'div'
     */
    tag?: keyof JSX.IntrinsicElements;
    tagProps?: object;
    style?: React.CSSProperties;
    textColor?: TextColor;
    textWeight?: TTextWeight;
    textSize?: TTextSize;
    inset?: TBoxSizes;
    x?: TBoxSizes;
    y?: TBoxSizes;
    /**
     * Отступ сверху
     */
    above?: TBoxSizes;
    /**
     * Отступ снизу
     */
    below?: TBoxSizes;
    between?: TBoxSizes;
    nowrap?: boolean;
    squish?: boolean;
    inline?: boolean;
    children?: React.ReactNode;
    innerRef?: React.Ref<HTMLElement> | Dispatch<SetStateAction<undefined>>;
    ref?: React.Ref<HTMLElement>;
    onClick?(e: React.MouseEvent<HTMLDivElement>): void;
}

const Box: React.FC<IBoxProps> = React.forwardRef<HTMLElement, IBoxProps>(
    (props, ref) => {
        const xSize = Number(props.inset || props.x);
        let ySize = Number(props.inset || props.y);

        if (ySize > 1 && props.squish) {
            ySize = Math.floor(ySize / 2);
        }

        const betweenClasses: string[] = [];

        if (props.between) {
            if (props.inline) {
                betweenClasses.push(
                    'x-between',
                    `x-between_space_${props.between}`,
                );
            } else {
                betweenClasses.push(`y-between_space_${props.between}`);
            }
        }

        const cn = cx(
            props.className,
            props.textColor && textColorCn(props.textColor),
            props.textSize && textSizeCn(props.textSize),
            props.textWeight && textWeightCn(props.textWeight),
            props.nowrap && 'nowrap',
            xSize && `x_${xSize}`,
            ySize && `y_${ySize}`,
            props.above && `above_${props.above}`,
            props.below && `below_${props.below}`,
            ...betweenClasses,
        );

        return React.createElement(
            props.tag || 'div',
            {
                ...props.tagProps,
                ...prepareQaAttributes(props),
                style: props.style,
                className: cn,
                ref: props.innerRef || ref,
                onClick: props.onClick,
            },
            props.children,
        );
    },
);

Box.displayName = 'Box';

export default Box;
