import React, {useCallback, useMemo} from 'react';

import {IWithClassName} from 'types/withClassName';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import isBreadcrumbsItem from 'components/Breadcrumbs/utilities/isBreadcrumbsItem';
import getMicrodataProps from 'components/Breadcrumbs/utilities/getMicrodataProps';

import {TBoxSizes} from 'components/Box/Box';
import BreadcrumbsItem, {
    IBreadcrumbsItemProps,
} from 'components/Breadcrumbs/components/BreadcrumbsItem/BreadcrumbsItem';
import Tabs from 'components/Tabs/Tabs';

interface IBreadcrumbsProps extends IWithClassName, IWithQaAttributes {
    /**
     * Дочерние элементы должны быть обёрнуты `<Breadcrumbs.Item />`.
     * Тогда они будут разделяться звёздочками
     */
    children: React.ReactNode;

    additional?: React.ReactNode;

    /**
     * Горизонтальный отступ. Такой же как проп "x" в `<Box />`
     */
    x?: TBoxSizes;

    /**
     * Включает отображение границы снизу
     */
    hasBorder?: boolean;

    /**
     * Добавляет микроразметку формата microdata
     */
    hasMicrodata?: boolean;
}

interface IBreadcrumbsStatic {
    Item: React.FC<IBreadcrumbsItemProps>;
}

interface ITab {
    id: number;
    child: React.ReactNode;
    disabled: boolean;
    active: boolean;
}

const Breadcrumbs: React.FC<IBreadcrumbsProps> & IBreadcrumbsStatic = props => {
    const {children, hasMicrodata = false, hasBorder, x, additional} = props;

    const tabs: ITab[] = React.Children.toArray(children).map((child, i) => {
        return {
            id: i,
            child,
            disabled: isBreadcrumbsItem(child)
                ? Boolean(child.props.disabled)
                : false,
            active: isBreadcrumbsItem(child)
                ? Boolean(child.props.active)
                : false,
        };
    });

    const getTabId = useCallback((tab: ITab) => {
        return tab.id;
    }, []);

    const getTabIsDisabled = useCallback((tab: ITab) => {
        return tab.disabled;
    }, []);

    const getTabIsActive = useCallback((tab: ITab) => {
        return tab.active;
    }, []);

    const activeTabId = useMemo(() => {
        const activeTab = tabs.find(getTabIsActive);

        if (!activeTab) {
            return undefined;
        }

        return getTabId(activeTab);
    }, [getTabId, getTabIsActive, tabs]);

    return (
        <Tabs
            arrow
            activeTabId={activeTabId}
            additional={additional}
            border={hasBorder}
            offset={x}
            tabs={tabs}
            renderTab={({tab: {child}, tabProps}): React.ReactNode => {
                if (!isBreadcrumbsItem(child)) {
                    return child;
                }

                return React.cloneElement(child, {innerRef: tabProps.innerRef});
            }}
            getTabId={getTabId}
            getTabIsDisabled={getTabIsDisabled}
            getTabIsActive={getTabIsActive}
            {...getMicrodataProps(hasMicrodata)}
            {...prepareQaAttributes(props)}
        />
    );
};

Breadcrumbs.Item = BreadcrumbsItem;

export default Breadcrumbs;
