import React, {useMemo} from 'react';
import {composeU as compose} from '@bem-react/core';
import {Link} from 'react-router-dom';
import classNames from 'classnames';

import {TLinkTarget} from 'types/common/TLinkTarget';

import {IButtonProps, getScaledIcon} from 'components/Button/Button';
import {ButtonPresenter} from 'components/Button/controls/ButtonPresenter';
import {withTypeLink} from '@yandex-lego/components/Button/desktop';

import {withRouterLink} from './_type/withTypeRouterLink';

const ButtonLinkPresenter = compose(
    withTypeLink,
    withRouterLink,
)(ButtonPresenter);

type TButtonLinkToType = React.ComponentProps<typeof Link>['to'];

export interface IButtonLinkProps extends Omit<IButtonProps, 'type' | 'value'> {
    /**
     * Адрес ссылки
     */
    url?: string;

    /**
     * Ссылка react-router
     */
    to?: TButtonLinkToType;

    /**
     * Аттрибут скачиваемого файла
     */
    download?: boolean | string;

    /**
     * Атрибут rel определяет отношения между текущим документом и документом, на который ведет ссылка, заданная атрибутом href
     */
    rel?: string;

    /**
     * Аттрибут target:
     *
     * - `_blank` — Загружает страницу в новое окно браузера.
     * - `_self` — Загружает страницу в текущее окно.
     * - `_parent` — Загружает страницу во фрейм-родитель, если фреймов нет, то это значение работает как `_self`.
     * - `_top` — Отменяет все фреймы и загружает страницу в полном окне браузера, если фреймов нет, то это значение работает как `_self`.
     *
     * @default '_self'
     */
    target?: TLinkTarget;
}

// eslint-disable-next-line valid-jsdoc
/**
 * Компонент для отрисовки обычных ссылок и ссылок react-router-dom,
 * которые выглядят как кнопка. Принимает все те же самые пропсы что и Button
 * за исключением type и value
 */
export default function ButtonLink({
    download,
    rel,
    target = '_self',
    to,
    url,
    disabled,
    size = 'm-inset',
    width = 'auto',
    theme = 'secondary',
    icon,
    iconLeft,
    iconRight,
    children,
    ...buttonProps
}: IButtonLinkProps) {
    const leftIcon = useMemo(() => {
        return (
            (icon || iconLeft) &&
            getScaledIcon(icon || iconLeft, size, !children, 'left')
        );
    }, [icon, iconLeft, size, children]);

    const rightIcon = useMemo(() => {
        return iconRight && getScaledIcon(iconRight, size, !children, 'right');
    }, [iconRight, size, children]);

    const buttonContent = useMemo(() => {
        return (
            <span className={classNames('YTButton-Text')}>
                {leftIcon}
                {children}
                {rightIcon}
            </span>
        );
    }, [children, rightIcon, leftIcon]);

    const commonProps = {
        ...buttonProps,
        children: buttonContent,
        view: 'default' as 'default',
        width: width === 'max' ? 'max' : undefined,
        tone: 'default' as 'default',
        size,
        theme,
    };

    if (disabled) {
        // @ts-ignore
        return <ButtonLinkPresenter disabled {...commonProps} />;
    }

    const linkProps = {
        download,
        rel,
        target,
    };

    if (to) {
        return (
            <ButtonLinkPresenter
                {...commonProps}
                {...linkProps}
                type="router-link"
                // @ts-ignore
                to={to}
            />
        );
    }

    return (
        // @ts-ignore
        <ButtonLinkPresenter
            {...commonProps}
            {...linkProps}
            type="link"
            url={url}
        />
    );
}
