import React, {useCallback, useMemo, useRef, useState} from 'react';
import {noop} from 'lodash';

import {TCalendarPricesMask} from 'components/Calendar/types';
import {IWithClassName} from 'types/withClassName';
import {TCalendarPrices} from 'types/avia/ICalendarPrice';

import {deviceMods} from 'utilities/stylesUtils';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import createCalendarMonths from 'components/Calendar/utilities/createCalendarMonths';
import createCalendarWeekdays from 'components/Calendar/utilities/createCalendarWeekdays';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import CalendarMonthsGrid from 'components/Calendar/components/CalendarMonthsGrid/CalendarMonthsGridNew';
import CalendarMonthsList from 'components/Calendar/components/CalendarMonthsList/CalendarMonthsListNew';

import cx from './Calendar.scss';

export interface ICalendarProps extends IWithClassName, IWithQaAttributes {
    calendarWrapperClassName?: string;
    dayClassName?: string;
    weekdaysClassName?: string;
    monthsGridClassName?: string;

    minDate: Date;
    maxDate: Date;
    nowDate: Date;
    startDate?: Nullable<Date>;
    endDate?: Nullable<Date>;
    weekStart?: number;
    siblingMonths?: boolean;
    footerNode?: React.ReactNode;
    headerNode?: React.ReactNode;
    isRenderMonthsList: boolean;
    cutPastWeeks?: boolean;
    canScrollIntoSelectedDate?: boolean;
    prices?: TCalendarPrices;
    mask?: TCalendarPricesMask;
    canSelectRange?: boolean;

    onDaySelect?: (data: {selectedDate: Date}) => void;
    onDayHover?: (date: Nullable<Date>) => void;

    onMonthsGridTouchStart?: (event: TouchEvent) => void;
    onMonthsGridTouchMove?: (event: TouchEvent) => void;
    onMonthsGridTouchEnd?: (event: TouchEvent) => void;
}

const CalendarNew: React.FC<ICalendarProps> = props => {
    const {
        className,
        calendarWrapperClassName,
        dayClassName,
        weekdaysClassName,
        monthsGridClassName,
        mask,
        prices,
        minDate,
        maxDate,
        nowDate,
        onDaySelect = noop,
        startDate = null,
        endDate = null,
        weekStart = 1,
        siblingMonths = false,
        footerNode = null,
        headerNode = null,
        isRenderMonthsList = true,
        cutPastWeeks = false,
        canScrollIntoSelectedDate = true,
        onMonthsGridTouchEnd,
        onMonthsGridTouchStart,
        onMonthsGridTouchMove,
        onDayHover,
        canSelectRange = false,
    } = props;
    const deviceType = useDeviceType();

    const [hoveredDate, setHoveredDate] = useState<null | Date>(null);
    const handleDayClick = useCallback(
        (selectedDate: Date) => {
            onDaySelect({
                selectedDate,
            });
        },
        [onDaySelect],
    );

    const handleSetHoveredDate = useCallback(
        (newValue: Nullable<Date>) => {
            if (deviceType.isMobile) {
                return;
            }

            setHoveredDate(newValue);
            onDayHover?.(newValue);
        },
        [onDayHover, setHoveredDate, deviceType.isMobile],
    );
    const handleDayMouseLeave = useCallback(() => {
        handleSetHoveredDate(null);
    }, [handleSetHoveredDate]);

    const months = useMemo(
        () =>
            createCalendarMonths({
                minDate,
                maxDate,
                weekStart,
                siblingMonths,
                cutPastWeeks,
                cutDate: nowDate,
            }),
        [minDate, maxDate, weekStart, siblingMonths, cutPastWeeks, nowDate],
    );
    const weekdays = useMemo(
        () => createCalendarWeekdays({weekStart}),
        [weekStart],
    );

    const isRangeSelected = useMemo(() => {
        if (!startDate || !canSelectRange) {
            return false;
        }

        if (endDate && Number(startDate) !== Number(endDate)) {
            return true;
        }

        return (
            hoveredDate !== null && Number(hoveredDate) !== Number(startDate)
        );
    }, [startDate, endDate, hoveredDate, canSelectRange]);

    const monthsGridRef = useRef<CalendarMonthsGrid>(null);
    const scrollToMonth = useCallback((index: number) => {
        if (!monthsGridRef.current) {
            return;
        }

        monthsGridRef.current.scrollToMonth(index);
    }, []);

    const [visibleMonth, setVisibleMonth] = useState(0);

    return (
        <div
            className={cx(
                'calendar',
                className,
                deviceMods('calendar', deviceType),
            )}
            {...prepareQaAttributes(props)}
        >
            {isRenderMonthsList && (
                <CalendarMonthsList
                    className={cx('monthsList', 'monthsList_new')}
                    months={months}
                    scrollToMonth={scrollToMonth}
                    highlightMonth={visibleMonth}
                />
            )}
            <div className={cx('calendarWrapper', calendarWrapperClassName)}>
                {headerNode && <div className={cx('header')}>{headerNode}</div>}
                <CalendarMonthsGrid
                    className={cx('monthsGrid', monthsGridClassName)}
                    ref={monthsGridRef}
                    dayClassName={dayClassName}
                    weekdaysClassName={weekdaysClassName}
                    months={months}
                    weekdays={weekdays}
                    canSelectRange={canSelectRange}
                    startDate={startDate ? startDate : undefined}
                    endDate={endDate ? endDate : undefined}
                    nowDate={nowDate}
                    minDate={minDate}
                    maxDate={maxDate}
                    isRangeSelected={Boolean(isRangeSelected)}
                    mask={mask}
                    deviceType={deviceType}
                    hoveredDate={hoveredDate ? hoveredDate : undefined}
                    onVisibleMonthChange={setVisibleMonth}
                    canScrollIntoSelectedDate={canScrollIntoSelectedDate}
                    prices={prices}
                    onDayClick={handleDayClick}
                    onDayMouseLeave={handleDayMouseLeave}
                    onDayMouseEnter={handleSetHoveredDate}
                    onTouchStart={onMonthsGridTouchStart}
                    onTouchMove={onMonthsGridTouchMove}
                    onTouchEnd={onMonthsGridTouchEnd}
                    {...prepareQaAttributes(props)}
                />
                {footerNode}
            </div>
        </div>
    );
};

export default CalendarNew;
