import React, {MouseEventHandler, PureComponent} from 'react';
import _noop from 'lodash/noop';

import {IWithClassName} from 'types/withClassName';
import {IWithDeviceType} from 'types/withDeviceType';
import {ICalendarPrice} from 'types/avia/ICalendarPrice';
import {ECalendarType, ICalendarDay} from 'components/Calendar/types';

import {deviceMods} from 'utilities/stylesUtils';
import {CURRENCY_RUB} from 'utilities/currency/codes';
import getUTCDateByParams from '../../utilities/getUTCDateByParams';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import Price from 'components/Price/Price';
import SearchIcon from 'icons/16/Search';

import cx from './CalendarDay.scss';

const searchIcon = (
    <SearchIcon className={cx('searchIcon')} width={12} height={12} />
);

interface ICalendarDayProps
    extends IWithClassName,
        IWithDeviceType,
        IWithQaAttributes {
    day: ICalendarDay | false;
    isRangeSelected: boolean;

    setScrolledDayNode: Function;
    onDayClick: Function;
    onDayMouseLeave: Function;
    onDayMouseEnter: Function;
    price?: ICalendarPrice | {unknown: true};
    setActiveDayRef: Function;

    isHovered: boolean;
    isStartDate: boolean;
    isEndDate: boolean;
    isScrolledDate: boolean;
    isRangeDay: boolean;
    isWeekend: boolean;
    isActive: boolean;
    isDisable: boolean;
}

class CalendarDay extends PureComponent<ICalendarDayProps> {
    private rootNode: HTMLDivElement | null = null;

    static readonly defaultProps: Partial<ICalendarDayProps> = {
        className: '',

        onDayClick: _noop,
        onDayMouseLeave: _noop,
        onDayMouseEnter: _noop,
        setScrolledDayNode: _noop,
        setActiveDayRef: _noop,

        isHovered: false,
        isStartDate: false,
        isEndDate: false,
        isScrolledDate: false,
        isRangeDay: false,
        isWeekend: false,
        isActive: false,
        isDisable: false,
    };

    componentDidMount(): void {
        this.setSelectedDayNode();
    }

    componentDidUpdate(prevProps: ICalendarDayProps): void {
        const {setActiveDayRef, isStartDate, isEndDate} = this.props;

        if (
            (prevProps.isStartDate !== isStartDate && isStartDate) ||
            (prevProps.isEndDate !== isEndDate && isEndDate)
        ) {
            setActiveDayRef(this.getRootNodeRef());
        }
    }

    private setSelectedDayNode(): void {
        const {setScrolledDayNode, isStartDate, isScrolledDate} = this.props;

        if (isScrolledDate) {
            const dayNode = this.getRootNodeRef();

            setScrolledDayNode({
                dayNode,
                type: isStartDate
                    ? ECalendarType.StartDate
                    : ECalendarType.EndDate,
            });
        }
    }

    private setRootNodeRef = (rootNode: HTMLDivElement): void => {
        this.rootNode = rootNode;
    };

    private getRootNodeRef = (): HTMLDivElement | null => {
        return this.rootNode;
    };

    private getDayParams(): Date {
        const {day} = this.props;

        // as: тк вызывается только при hover на существующий день
        // (непустая клетка календаря в таблице месяце)
        return getUTCDateByParams(day as ICalendarDay);
    }

    private handleMouseLeave: MouseEventHandler = e => {
        const {onDayMouseLeave} = this.props;
        const day = this.getDayParams();

        onDayMouseLeave(day, e);
    };

    private handleMouseEnter: MouseEventHandler = () => {
        const {onDayMouseEnter} = this.props;
        const day = this.getDayParams();

        onDayMouseEnter(day);
    };

    private handleClick: MouseEventHandler = e => {
        const {onDayClick} = this.props;
        const day = this.getDayParams();

        onDayClick(day, e);
    };

    private renderEmptyDay(): React.ReactNode {
        const {className, deviceType} = this.props;

        return (
            <div
                className={cx(
                    'day',
                    'day_empty_yes',
                    deviceMods('day', deviceType),
                    className,
                )}
            />
        );
    }

    private renderDay({day, month, year}: ICalendarDay): React.ReactNode {
        const {
            className,
            deviceType,
            isWeekend,
            isHovered,
            isStartDate,
            isEndDate,
            isRangeDay,
            isActive,
            isDisable,
            isRangeSelected,
        } = this.props;

        return (
            <div
                ref={this.setRootNodeRef}
                className={cx(
                    'day',
                    {
                        day_weekend_yes: isWeekend,
                        day_hover_yes: isHovered,
                        day_start_yes: isStartDate,
                        day_startWithContinue: isRangeSelected,
                        day_end_yes: isEndDate,
                        day_range_yes: isRangeDay,
                        day_active: isActive,
                        day_disable_yes: isDisable,
                    },
                    deviceMods('day', deviceType),
                    className,
                )}
                onMouseEnter={this.handleMouseEnter}
                onMouseLeave={this.handleMouseLeave}
                onClick={this.handleClick}
                {...prepareQaAttributes({
                    parent: this.props,
                    current: `day-${year}-${String(month + 1).padStart(
                        2,
                        '0',
                    )}-${String(day).padStart(2, '0')}`,
                })}
            >
                <span>{day}</span>
                {this.renderPrice()}
            </div>
        );
    }

    private renderPrice(): React.ReactNode {
        const {isDisable, price} = this.props;

        if (!price) {
            return null;
        }

        if (isDisable) {
            return <div className={cx('price')} />;
        }

        return (
            <div
                className={cx('price', {
                    price_withValue: !('unknown' in price),
                    price_isLow: !('unknown' in price) && price.isLow,
                })}
            >
                {'unknown' in price ? (
                    searchIcon
                ) : (
                    <Price
                        isRound
                        isRoughly={price.roughly}
                        value={price.value}
                        currency={CURRENCY_RUB}
                    />
                )}
            </div>
        );
    }

    render(): React.ReactNode {
        const {day} = this.props;

        return day ? this.renderDay(day) : this.renderEmptyDay();
    }
}

export default CalendarDay;
