import React, {MouseEventHandler, PureComponent} from 'react';
import _noop from 'lodash/noop';

import {ICalendarDay} from 'components/Calendar/types';
import {IWithClassName} from 'types/withClassName';
import {ICalendarPrice} from 'types/avia/ICalendarPrice';

import {IDevice} from 'reducers/common/commonReducerTypes';

import {CURRENCY_RUB} from 'utilities/currency/codes';
import getUTCDateByParams from '../../utilities/getUTCDateByParams';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {deviceMods} from 'utilities/stylesUtils';
import getCalendarDayHeight from 'components/Calendar/utilities/getCalendarDayHeight';

import Price from 'components/Price/Price';
import SearchIcon from 'icons/12/Search';

import cx from './CalendarDayNew.scss';

interface ICalendarDayProps extends IWithClassName, IWithQaAttributes {
    day: ICalendarDay | false;
    isRangeSelected: boolean;
    deviceType: IDevice;

    setScrolledDayNode: Function;
    onDayClick: Function;
    onDayMouseLeave: Function;
    onDayMouseEnter: Function;
    price?: ICalendarPrice | {unknown: true};
    setActiveDayRef: Function;

    isHovered: boolean;
    isStartDate: boolean;
    isEndDate: boolean;
    isScrolledDate: boolean;
    isRangeDay: boolean;
    isWeekend: boolean;
    isActive: boolean;
    isDisable: boolean;
}

class CalendarDayNew extends PureComponent<ICalendarDayProps> {
    private rootNode: HTMLDivElement | null = null;

    static defaultProps = {
        className: '',

        onDayClick: _noop,
        onDayMouseLeave: _noop,
        onDayMouseEnter: _noop,
        setScrolledDayNode: _noop,
        setActiveDayRef: _noop,

        isHovered: false,
        isStartDate: false,
        isEndDate: false,
        isScrolledDate: false,
        isRangeDay: false,
        isWeekend: false,
        isActive: false,
        isDisable: false,
    };

    componentDidMount(): void {
        this.setSelectedDayNode();
    }

    componentDidUpdate(prevProps: ICalendarDayProps): void {
        const {setActiveDayRef, isStartDate, isEndDate} = this.props;

        if (
            (prevProps.isStartDate !== isStartDate && isStartDate) ||
            (prevProps.isEndDate !== isEndDate && isEndDate)
        ) {
            setActiveDayRef(this.getRootNodeRef());
        }
    }

    setSelectedDayNode(): void {
        const {setScrolledDayNode, isScrolledDate} = this.props;

        if (isScrolledDate) {
            const dayNode = this.getRootNodeRef();

            setScrolledDayNode({
                dayNode,
            });
        }
    }

    setRootNodeRef = (rootNode: HTMLDivElement): void => {
        this.rootNode = rootNode;
    };

    getRootNodeRef = (): HTMLDivElement | null => {
        return this.rootNode;
    };

    getDayParams(): Date {
        const {day} = this.props;

        // as: тк вызывается только при hover на существующий день
        // (непустая клетка календаря в таблице месяце)
        return getUTCDateByParams(day as ICalendarDay);
    }

    handleMouseLeave: MouseEventHandler = e => {
        const {onDayMouseLeave} = this.props;
        const day = this.getDayParams();

        onDayMouseLeave(day, e);
    };

    handleMouseEnter: MouseEventHandler = () => {
        const {onDayMouseEnter} = this.props;
        const day = this.getDayParams();

        onDayMouseEnter(day);
    };

    handleClick: MouseEventHandler = e => {
        const {onDayClick} = this.props;
        const day = this.getDayParams();

        onDayClick(day, e);
    };

    renderEmptyDay(): React.ReactNode {
        const {className, deviceType} = this.props;

        return (
            <div
                className={cx(
                    'day',
                    'day_empty_yes',
                    deviceMods('day', deviceType),
                    className,
                )}
            />
        );
    }

    renderDay({day, month, year}: ICalendarDay): React.ReactNode {
        const {
            className,
            deviceType,
            isWeekend,
            isHovered,
            isStartDate,
            isEndDate,
            isRangeDay,
            isActive,
            isDisable,
            isRangeSelected,
        } = this.props;

        return (
            <div
                ref={this.setRootNodeRef}
                className={cx(
                    'day',
                    {
                        day_type_weekend: isWeekend,
                        day_type_hover: isHovered,
                        day_type_start: isStartDate,
                        day_type_startWithContinue: isRangeSelected,
                        day_type_end: isEndDate,
                        day_type_inRange: isRangeDay,
                        day_type_active: isActive,
                        day_type_disabled: isDisable,
                    },
                    deviceMods('day', deviceType),
                    className,
                )}
                onMouseEnter={this.handleMouseEnter}
                onMouseLeave={this.handleMouseLeave}
                style={{height: getCalendarDayHeight(deviceType.isMobile)}}
                onClick={this.handleClick}
                {...prepareQaAttributes({
                    parent: this.props,
                    current: `day-${year}-${String(month + 1).padStart(
                        2,
                        '0',
                    )}-${String(day).padStart(2, '0')}`,
                })}
            >
                <span>{day}</span>
                {this.renderPrice()}
            </div>
        );
    }

    renderPrice(): React.ReactNode {
        const {isDisable, price} = this.props;

        if (!price) {
            return null;
        }

        if (isDisable) {
            return <div className={cx('price')} />;
        }

        if ('unknown' in price) {
            return (
                <div className={cx('price')}>
                    <SearchIcon className={cx('searchIcon')} />
                </div>
            );
        }

        return (
            <div
                className={cx('price', 'price_withValue', {
                    price_isLow: price.isLow,
                })}
            >
                <Price
                    isRound
                    isRoughly={price.roughly}
                    value={price.value}
                    currency={CURRENCY_RUB}
                />
            </div>
        );
    }

    render(): React.ReactNode {
        const {day} = this.props;

        return day ? this.renderDay(day) : this.renderEmptyDay();
    }
}

export default CalendarDayNew;
