import React, {PureComponent} from 'react';
import _noop from 'lodash/noop';

import {IWithClassName} from 'types/withClassName';
import {
    ICalendarDay,
    ICalendarMonth,
    TCalendarMask,
} from 'components/Calendar/types';
import {ICalendarPrice, TCalendarPrices} from 'types/avia/ICalendarPrice';

import {IDevice} from 'reducers/common/commonReducerTypes';

import getMonthWeeks from '../../utilities/getMonthWeeks';
import getCalendarDaySettings, {
    TCalendarDaySettings,
} from '../../utilities/getCalendarDaySettings';
import getISObyDate from '../../utilities/getISObyDate';
import getUTCDateByParams from '../../utilities/getUTCDateByParams';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import CalendarDay from '../CalendarDay/CalendarDayNew';

import cx from './CalendarMonth.scss';

interface ICalendarMonthProps extends IWithClassName, IWithQaAttributes {
    deviceType: IDevice;
    dayClassName: string;
    month: ICalendarMonth;
    nowDate: Date;
    minDate: Date;
    maxDate: Date;
    startDate: Date;
    endDate: Date;
    hoveredDate: Date;
    isRangeSelected: boolean;
    showLabel: boolean;
    canSelectRange?: boolean;
    setActiveDayRef?: Function;
    setScrolledDayNode: Function;
    prices?: TCalendarPrices;
    mask?: TCalendarMask;
    onDayClick: Function;
    onDayMouseLeave: Function;
    onDayMouseEnter: Function;
}

class CalendarMonth extends PureComponent<ICalendarMonthProps> {
    static defaultProps = {
        className: '',
        dayClassName: '',
        month: {},
        startDate: null,
        endDate: null,
        hoveredDate: false,
        showLabel: true,
        setScrolledDayNode: _noop,
        onDayClick: _noop,
        onDayMouseLeave: _noop,
        onDayMouseEnter: _noop,
    };

    getPriceForDay(
        day: ICalendarDay | false,
    ): ICalendarPrice | {unknown: true} | undefined {
        const {prices} = this.props;

        if (!prices || !day) {
            return undefined;
        }

        const price = prices[getISObyDate(getUTCDateByParams(day))];

        if (!price) {
            return {unknown: true};
        }

        return price;
    }

    getDaySettings(day: ICalendarDay | false): TCalendarDaySettings {
        const {
            mask,
            startDate,
            endDate,
            nowDate,
            minDate,
            maxDate,
            hoveredDate,
            canSelectRange,
        } = this.props;

        return getCalendarDaySettings({
            day,
            mask,
            startDate,
            endDate,
            nowDate,
            minDate,
            maxDate,
            hoveredDate,
            canSelectRange,
            newExperimentalCalendar: true,
        });
    }

    renderMonthDay = (
        day: false | ICalendarDay,
        dayIndex: number,
    ): React.ReactNode => {
        const {
            deviceType,
            dayClassName,
            isRangeSelected,
            onDayClick,
            onDayMouseLeave,
            onDayMouseEnter,
            setActiveDayRef,
            setScrolledDayNode,
        } = this.props;

        const daySettings = this.getDaySettings(day);

        return (
            <CalendarDay
                className={dayClassName}
                deviceType={deviceType}
                day={day}
                key={dayIndex}
                price={this.getPriceForDay(day)}
                isRangeSelected={isRangeSelected}
                onDayClick={onDayClick}
                onDayMouseLeave={onDayMouseLeave}
                onDayMouseEnter={onDayMouseEnter}
                setScrolledDayNode={setScrolledDayNode}
                setActiveDayRef={setActiveDayRef}
                {...daySettings}
                {...prepareQaAttributes(this.props)}
            />
        );
    };

    renderMonthWeek = (
        week: (false | ICalendarDay)[],
        weekIndex: number,
    ): React.ReactNode => {
        return (
            <div
                className={cx('week')}
                key={weekIndex}
                {...prepareQaAttributes('calendar-month-week')}
            >
                {week.map(this.renderMonthDay)}
            </div>
        );
    };

    renderMonthLabel(): React.ReactNode {
        const {month, showLabel} = this.props;

        if (!showLabel) {
            return null;
        }

        return (
            <div
                className={cx('month', 'month_new')}
                {...prepareQaAttributes('calendar-month-text')}
            >
                {month.monthLabel}
            </div>
        );
    }

    renderMonthDays = (): React.ReactNode => {
        const {
            month: {monthDays},
        } = this.props;
        const monthWeeks = getMonthWeeks(monthDays);

        return monthWeeks.map(this.renderMonthWeek);
    };

    render(): React.ReactNode {
        const {className} = this.props;

        return (
            <div
                className={cx('calendarMonth', className)}
                {...prepareQaAttributes('calendar-month')}
            >
                {this.renderMonthLabel()}
                {this.renderMonthDays()}
            </div>
        );
    }
}

export default CalendarMonth;
