import React, {PureComponent} from 'react';
import _noop from 'lodash/noop';
import {subscribe, unsubscribe} from 'pubsub-js';
import {InView} from 'react-intersection-observer';

import {CALENDAR_EVENTS} from 'components/Calendar/constants/constants';

import {IWithClassName} from 'types/withClassName';
import {
    ICalendarMonth,
    TCalendarMask,
    TWeekDayObject,
} from 'components/Calendar/types';
import {TCalendarPrices} from 'types/avia/ICalendarPrice';

import {IDevice} from 'reducers/common/commonReducerTypes';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {deviceModDesktop} from 'utilities/stylesUtils';
import sortDates from 'utilities/dateUtils/sortDates';
import getCalendarDayHeight from 'components/Calendar/utilities/getCalendarDayHeight';

import CalendarMonth from 'components/Calendar/components/CalendarMonth/CalendarMonthNew';
import CalendarWeekdays from 'components/Calendar/components/CalendarWeekdays/CalendarWeekdays';
import CalendarScrollController from 'components/Calendar/components/CalendarScrollController/CalendarScrollController';

import cx from './CalendarMonthsGridNew.scss';

const MIN_MONTH_TO_RENDER = 3;
const MONTH_TITLE_HEIGHT = 32;

interface ICalendarMonthsGridProps extends IWithClassName, IWithQaAttributes {
    deviceType: IDevice;
    dayClassName: string;
    weekdaysClassName: string;
    months: ICalendarMonth[];
    weekdays: TWeekDayObject[];
    nowDate: Date;
    minDate: Date;
    maxDate: Date;
    startDate: Date;
    endDate: Date;
    hoveredDate?: Date;
    isRangeSelected: boolean;
    canScrollIntoSelectedDate: boolean;
    canSelectRange?: boolean;
    prices?: TCalendarPrices;
    mask?: TCalendarMask;
    onDayClick: Function;
    onDayMouseLeave: Function;
    onDayMouseEnter: Function;

    onTouchStart?: (event: TouchEvent) => void;
    onTouchMove?: (event: TouchEvent) => void;
    onTouchEnd?: (event: TouchEvent) => void;
    onVisibleMonthChange?: (index: number) => void;
}

interface ICalendarMonthsGridState {
    animationInProgress: boolean;
    scrollPosition: number;
    lastScrollToMonth: number;
}

class CalendarMonthsGrid extends PureComponent<
    ICalendarMonthsGridProps,
    ICalendarMonthsGridState
> {
    private monthsListNode: HTMLDivElement | null = null;
    private scrolledDayNode: HTMLDivElement | null = null;
    private activeDayRef: HTMLDivElement | null = null;

    static defaultProps = {
        className: '',
        dayClassName: '',
        weekdaysClassName: '',
        months: [],
        weekdays: [],
        onChangeScrollPosition: _noop,
        startDate: null,
        endDate: null,
        headerNode: null,
        canScrollIntoSelectedDate: true,
        onDayClick: _noop,
        onDayMouseLeave: _noop,
        onDayMouseEnter: _noop,
    };

    constructor(props: ICalendarMonthsGridProps) {
        super(props);

        this.state = {
            animationInProgress: false,
            scrollPosition: 0,
            lastScrollToMonth: 0,
        };
    }

    componentDidMount(): void {
        this.subscribe();
        this.scrollIntoScrolledDay();

        this.addTouchEventListeners();
    }

    componentWillUnmount(): void {
        this.unsubscribe();
        this.removeTouchEventListeners();
    }

    subscribe(): void {
        subscribe(
            CALENDAR_EVENTS.SCROLL_TO_ACTIVE_DATE,
            this.scrollToActiveDate,
        );
    }

    unsubscribe(): void {
        unsubscribe(CALENDAR_EVENTS.SCROLL_TO_ACTIVE_DATE);
    }

    addTouchEventListeners(): void {
        const ref = this.getMonthsListRef();

        if (!ref) {
            return;
        }

        const {onTouchStart, onTouchEnd, onTouchMove} = this.props;

        // Добавляем через eventListeners, вместо системы событий реакта,
        // Так как нет возможности указать {passive: false}
        if (onTouchStart) {
            ref.addEventListener('touchstart', onTouchStart);
        }

        if (onTouchMove) {
            ref.addEventListener('touchmove', onTouchMove, {passive: false});
        }

        if (onTouchEnd) {
            ref.addEventListener('touchend', onTouchEnd);
        }
    }

    removeTouchEventListeners(): void {
        const ref = this.getMonthsListRef();

        if (!ref) {
            return;
        }

        const {onTouchStart, onTouchEnd, onTouchMove} = this.props;

        if (onTouchStart) {
            ref.removeEventListener('touchstart', onTouchStart);
        }

        if (onTouchMove) {
            ref.removeEventListener('touchmove', onTouchMove);
        }

        if (onTouchEnd) {
            ref.removeEventListener('touchend', onTouchEnd);
        }
    }

    /* Refs manipulate */

    setMonthsListRef = (monthListNode: HTMLDivElement): void => {
        this.monthsListNode = monthListNode;
    };

    getMonthsListRef = (): HTMLDivElement | null => {
        return this.monthsListNode;
    };

    setScrolledDayNodeRef = (dayNode: HTMLDivElement | null): void => {
        this.scrolledDayNode = dayNode;
    };

    getScrolledDayNodeRef = (): HTMLDivElement | null => {
        return this.scrolledDayNode;
    };

    setActiveDayRef = (node: HTMLDivElement | null): void => {
        this.activeDayRef = node;
    };

    getActiveDayRef = (): HTMLDivElement | null => {
        return this.activeDayRef;
    };

    getMonthHeight = (index: number): number => {
        const {months, deviceType} = this.props;
        const dayHeight = getCalendarDayHeight(deviceType.isMobile);
        const month = months[index];
        const labelHeight = MONTH_TITLE_HEIGHT;
        const rowsCount = month.monthDays.length / 7;

        return rowsCount * dayHeight + labelHeight;
    };

    // Не используем hoveredDate напрямую из props, чтобы не вызывать лишних
    // перерендеров у компонентов, на которые не влияет изменение hoveredDate
    getHoveredDate(monthNum: ICalendarMonth['month']): undefined | Date {
        const {hoveredDate, startDate, endDate} = this.props;

        if (!hoveredDate || hoveredDate.getMonth() === monthNum) {
            return hoveredDate;
        }

        // При выделении диапазона нам необходимо перерисовать все месяцы,
        // которые попадают в диапазон
        if (startDate && !endDate) {
            const [left, right] = sortDates(startDate, hoveredDate);

            return monthNum <= right.getMonth() && monthNum >= left.getMonth()
                ? hoveredDate
                : undefined;
        }

        return undefined;
    }

    scrollIntoScrolledDay(): void {
        const {canScrollIntoSelectedDate} = this.props;
        const scrolledDayNode = this.getScrolledDayNodeRef();

        if (scrolledDayNode && canScrollIntoSelectedDate) {
            this.scrollToNode(scrolledDayNode);
        }
    }

    scrollToActiveDate = (): void => {
        const activeDayNode = this.getActiveDayRef();

        if (activeDayNode) {
            this.scrollToNode(activeDayNode);
        }
    };

    private getScrollToNodeOffset = (node: HTMLDivElement): number => {
        const monthsNode = this.getMonthsListRef();

        if (!node || !monthsNode) {
            return 0;
        }

        const nodePosition = node.getBoundingClientRect();
        const monthsPosition = monthsNode.getBoundingClientRect();
        const offsetTop =
            nodePosition.top - (monthsPosition.top - monthsNode.scrollTop) || 0;

        return offsetTop;
    };

    scrollToMonth = (index: number): void => {
        const monthsNode = this.getMonthsListRef();

        if (!monthsNode) {
            return;
        }

        const node = monthsNode.children[index] as HTMLDivElement;
        const offset = this.getScrollToNodeOffset(node);

        this.setState({
            animationInProgress: true,
            scrollPosition: offset,
            lastScrollToMonth: index,
        });
    };

    scrollToNode(node: HTMLDivElement): void {
        /* При монтировании компонента у списка месяцев в первый момент
         * offsetHeight и scrollHeight равны, что приводит к maxScrollTop=Infinity
         * rAF для вывода из синхронного потока
         */
        requestAnimationFrame(() => {
            const monthsNode = this.getMonthsListRef();

            if (node && monthsNode) {
                monthsNode.scrollTop =
                    this.getScrollToNodeOffset(node) - MONTH_TITLE_HEIGHT;
            }
        });
    }

    changeScrolledDayNode = ({dayNode}: {dayNode: HTMLDivElement}): void => {
        const scrolledDayNode = this.getScrolledDayNodeRef();

        if (!scrolledDayNode) {
            this.setScrolledDayNodeRef(dayNode);
        }
    };

    renderWeekdays(): React.ReactNode {
        const {weekdays, weekdaysClassName} = this.props;

        return (
            <CalendarWeekdays
                className={cx('weekdays', weekdaysClassName)}
                weekdays={weekdays}
            />
        );
    }

    renderMonthOrPlaceholder = (
        month: ICalendarMonth,
        index: number,
    ): React.ReactNode => {
        const {lastScrollToMonth} = this.state;
        const {startDate, endDate, nowDate} = this.props;
        const monthsShouldBeRenderUpTo = Math.max(
            ...[
                startDate
                    ? (startDate.getFullYear() - nowDate.getFullYear()) * 12 +
                      startDate.getMonth() +
                      1
                    : 0,
                endDate
                    ? (endDate.getFullYear() - nowDate.getFullYear()) * 12 +
                      endDate.getMonth() +
                      1
                    : 0,
                MIN_MONTH_TO_RENDER,
                lastScrollToMonth,
            ],
        );

        return (
            <InView key={index} triggerOnce>
                {({ref, inView}): React.ReactNode => (
                    <div ref={ref}>
                        {inView || index < monthsShouldBeRenderUpTo
                            ? this.renderMonth(index)
                            : this.renderPlaceholder(index)}
                    </div>
                )}
            </InView>
        );
    };

    renderPlaceholder = (index: number): React.ReactNode => (
        <div
            className={cx('monthPlaceholder')}
            style={{height: this.getMonthHeight(index)}}
        />
    );

    renderMonth = (index: number): React.ReactNode => {
        const {
            deviceType,
            mask,
            months,
            startDate,
            endDate,
            nowDate,
            minDate,
            maxDate,
            isRangeSelected,
            prices,
            dayClassName,
            onDayClick,
            onDayMouseLeave,
            onDayMouseEnter,
            canSelectRange,
            onVisibleMonthChange,
        } = this.props;

        return (
            <InView
                as="div"
                threshold={0.5}
                onChange={(inView): void => {
                    if (!inView) {
                        return;
                    }

                    onVisibleMonthChange?.(index);
                }}
            >
                <CalendarMonth
                    className={cx('month')}
                    deviceType={deviceType}
                    month={months[index]}
                    mask={mask}
                    startDate={startDate}
                    endDate={endDate}
                    nowDate={nowDate}
                    minDate={minDate}
                    maxDate={maxDate}
                    dayClassName={dayClassName}
                    hoveredDate={this.getHoveredDate(months[index].month)}
                    isRangeSelected={isRangeSelected}
                    onDayClick={onDayClick}
                    onDayMouseLeave={onDayMouseLeave}
                    onDayMouseEnter={onDayMouseEnter}
                    setActiveDayRef={this.setActiveDayRef}
                    canSelectRange={canSelectRange}
                    setScrolledDayNode={this.changeScrolledDayNode}
                    prices={prices}
                    {...prepareQaAttributes(this.props)}
                />
            </InView>
        );
    };

    renderMonths(): React.ReactNode {
        const {months} = this.props;

        return (
            <div className={cx('months')} ref={this.setMonthsListRef}>
                {months.map(this.renderMonthOrPlaceholder)}
            </div>
        );
    }

    onAnimationEnd = (): void => {
        this.setState({animationInProgress: false});
    };

    renderCalendarScrollController(): React.ReactNode {
        const {scrollPosition, animationInProgress} = this.state;
        const monthsListNode = this.getMonthsListRef();

        return (
            <CalendarScrollController
                scrollTop={scrollPosition}
                scrollNode={monthsListNode}
                onSpringRest={this.onAnimationEnd}
                withoutAnimation={!animationInProgress}
            />
        );
    }

    render(): React.ReactNode {
        const {className, deviceType} = this.props;

        return (
            <div
                className={cx(
                    'root',
                    deviceModDesktop('root', deviceType),
                    className,
                )}
            >
                {this.renderWeekdays()}
                {this.renderMonths()}
                {this.renderCalendarScrollController()}
            </div>
        );
    }
}

export default CalendarMonthsGrid;
