import React, {PureComponent} from 'react';
import {Spring, animated} from 'react-spring';

import {IWithClassName} from 'types/withClassName';
import {ICalendarMonth} from 'components/Calendar/types';

import cx from './CalendarMonthsListNew.scss';

const ACTIVE_MONTHS_SCROLL_HEIGHT = 32;
const ACTIVE_MONTHS_SCROLL_OFFSET = 8;

interface ICalendarMonthsListProps extends IWithClassName {
    months: ICalendarMonth[];
    highlightMonth: number;
    scrollToMonth: (index: number) => void;
}

interface ICalendarMonthsListState {
    monthsListHeight: number;
}

type TMonthsListNode = HTMLDivElement | null;

class CalendarMonthsListNew extends PureComponent<
    ICalendarMonthsListProps,
    ICalendarMonthsListState
> {
    static defaultProps = {
        className: '',
        months: [],
        highlightMonth: 0,
    };
    monthsListNode: TMonthsListNode = null;

    constructor(props: ICalendarMonthsListProps) {
        super(props);

        this.state = {
            monthsListHeight: 0,
        };
    }

    componentDidMount(): void {
        this.calculateAndSetMonthsListHeight();
    }

    calculateAndSetMonthsListHeight(): void {
        const monthsListNode = this.getMonthsListRef();

        if (!monthsListNode) {
            return;
        }

        const {height: monthsListHeight} =
            monthsListNode.getBoundingClientRect();

        this.setState({monthsListHeight});
    }

    getMonthsScrollPositionProps(): number {
        const {monthsListHeight} = this.state;
        const {highlightMonth, months} = this.props;

        const step = monthsListHeight / months.length;

        return step * highlightMonth;
    }

    setMonthsListRef = (monthsListNode: TMonthsListNode): void => {
        this.monthsListNode = monthsListNode;
    };

    getMonthsListRef = (): TMonthsListNode => {
        return this.monthsListNode;
    };

    renderActiveMonthsScrollPosition(): React.ReactNode {
        const scrollTopPosition = this.getMonthsScrollPositionProps();

        return (
            <Spring to={{transform: `translateY(${scrollTopPosition}px)`}}>
                {({transform}): React.ReactElement => (
                    <animated.div
                        className={cx('activeMonthsScroll')}
                        style={{
                            top: ACTIVE_MONTHS_SCROLL_OFFSET,
                            height: ACTIVE_MONTHS_SCROLL_HEIGHT,
                            transform,
                        }}
                    />
                )}
            </Spring>
        );
    }

    renderMonth = (
        {monthLabel, month, year}: ICalendarMonth,
        monthIndex: number,
    ): React.ReactNode => {
        const isFirstMonth = month === 0;
        const {scrollToMonth} = this.props;

        return (
            <div
                key={monthIndex}
                className={cx('month')}
                onClick={(): void => scrollToMonth(monthIndex)}
            >
                {monthLabel}
                {isFirstMonth && ` ${year}`}
            </div>
        );
    };

    renderMonthsList(): React.ReactNode {
        const {months} = this.props;

        return (
            <div ref={this.setMonthsListRef} className={cx('monthsList')}>
                {months.map(this.renderMonth)}
            </div>
        );
    }

    render(): React.ReactNode {
        const {className} = this.props;

        return (
            <div className={cx('calendarMonthsList', className)}>
                {this.renderActiveMonthsScrollPosition()}
                {this.renderMonthsList()}
            </div>
        );
    }
}

export default CalendarMonthsListNew;
