import {ISize} from 'types/common/TImage';

const DEFAULT_IMAGE_WIDTH = 200;

export function getInitialOffsetByIndex(
    container: HTMLDivElement | null,
    items: (HTMLDivElement | null)[] = [],
    initialVisibleIndex?: number,
): number {
    if (!container) {
        return 0;
    }

    if (typeof initialVisibleIndex === 'undefined') {
        return 0;
    }

    const lastItem = items[items.length - 1];
    const initialItem = items[initialVisibleIndex];

    if (!lastItem || !initialItem) {
        return 0;
    }

    const containerRect = container.getBoundingClientRect();
    const lastItemRect = lastItem.getBoundingClientRect();
    const initialItemRect = initialItem.getBoundingClientRect();

    const containerWidth = containerRect.right - containerRect.left;

    if (lastItemRect.right - initialItemRect.left > containerWidth) {
        return initialItemRect.left - containerRect.left;
    }

    return lastItemRect.right - containerRect.left - containerWidth;
}

/**
 * Check that the last element right boundary is not to the right of the container right boundary
 * Offset is calculated when the carousel is first rendered and isn't applied to DOM yet
 * @param container
 * @param items
 * @param initialOffset
 */
export function getInitialHasNextByOffset(
    container: HTMLDivElement | null,
    items: (HTMLDivElement | null)[] = [],
    initialOffset: number,
): boolean {
    if (!container) {
        return false;
    }

    const lastItem = items[items.length - 1];

    if (!lastItem) {
        return false;
    }

    const containerRect = container.getBoundingClientRect();
    const lastItemRect = lastItem.getBoundingClientRect();

    if (lastItemRect.right - initialOffset > containerRect.right) {
        return true;
    }

    return false;
}

/**
 * Check that the first element left boundary is not to the left of the container left boundary
 * Offset is calculated when the carousel is first rendered and isn't applied to DOM yet
 * @param container
 * @param items
 * @param initialOffset
 */
export function getInitialHasPrevByOffset(
    container: HTMLDivElement | null,
    items: (HTMLDivElement | null)[] = [],
    initialOffset: number,
): boolean {
    if (!container) {
        return false;
    }

    const firstItem = items[0];

    if (!firstItem) {
        return false;
    }

    const containerRect = container.getBoundingClientRect();
    const firstItemRect = firstItem.getBoundingClientRect();

    if (firstItemRect.left - initialOffset < containerRect.left) {
        return true;
    }

    return false;
}

/**
 * Возвращает индекс первого элемента, который не виден полностью в карусели слева
 */
export function getInvisibleLeftIndex(
    container: HTMLDivElement | null,
    items: (HTMLDivElement | null)[] = [],
): number | undefined {
    if (!container) {
        return;
    }

    const containerRect = container.getBoundingClientRect();

    for (let i = items.length - 1; i >= 0; i--) {
        const item = items[i];

        if (item) {
            const itemRect = item.getBoundingClientRect();

            if (Math.ceil(itemRect.left) < Math.floor(containerRect.left)) {
                return i;
            }
        }
    }
}

/**
 * Возвращает индекс первого элемента, который не виден полностью в карусели справа
 */
export function getInvisibleRightIndex(
    container: HTMLDivElement | null,
    items: (HTMLDivElement | null)[] = [],
): number | undefined {
    if (!container) {
        return;
    }

    const containerRect = container.getBoundingClientRect();

    for (let i = 0, length = items.length; i < length; i++) {
        const item = items[i];

        if (item) {
            const itemRect = item.getBoundingClientRect();

            if (Math.ceil(containerRect.right) < Math.floor(itemRect.right)) {
                return i;
            }
        }
    }
}

/**
 * Возвращает индекс правого элемента, который не виден полностью в карусели справа, на основе размеров элементов
 */
export function getInvisibleRightIndexBySizes(
    container: HTMLDivElement | null,
    lazyLoadSizes: (ISize | undefined)[],
) {
    if (container) {
        const containerRect = container.getBoundingClientRect();

        const {width: containerWidth, height: containerHeight} = containerRect;

        if (containerWidth) {
            let imagesWidth = 0;
            let count = 1;

            for (let i = 0; i < lazyLoadSizes.length; i++) {
                const size = lazyLoadSizes[i];
                const aspectRatio =
                    (size && containerHeight / size.height) || 1;

                imagesWidth +=
                    (size && size.width * aspectRatio) || DEFAULT_IMAGE_WIDTH;
                count++;

                if (imagesWidth > containerWidth) {
                    return count;
                }
            }
        }
    }
}
