import React, {FunctionComponent, useMemo} from 'react';
import {composeU as compose} from '@bem-react/core';
import classNames from 'classnames';

import {proxyRefCb} from 'utilities/proxyRefCb';

import {ButtonPresenter} from 'components/Button/controls/ButtonPresenter';
import {IButtonProps, getScaledIcon} from 'components/Button/Button';

import {withKindCheck} from './_type/withKindCheck';

import cx from './CheckButton.scss';

const CheckButtonPresenter = compose(withKindCheck)(ButtonPresenter);

export interface ICheckButtonProps
    extends Omit<IButtonProps, 'value' | 'type'> {
    /**
     * Значение кнопки
     */
    checked: boolean;
}

const CheckButton: FunctionComponent<ICheckButtonProps> = props => {
    const {size} = props;
    const {
        width,
        children,
        icon,
        iconLeft,
        iconRight,
        innerRef,
        className,
        ...rest
    } = props;

    const leftIcon = useMemo(() => {
        return (
            (icon || iconLeft) &&
            getScaledIcon(icon || iconLeft, size, !children, 'left')
        );
    }, [icon, iconLeft, size, children]);

    const rightIcon = useMemo(() => {
        return iconRight && getScaledIcon(iconRight, size, !children, 'right');
    }, [iconRight, size, children]);

    const buttonContent = useMemo(() => {
        return (
            <span className={classNames('YTButton-Text')}>
                {leftIcon}
                {children}
                {rightIcon}
            </span>
        );
    }, [children, rightIcon, leftIcon]);

    return (
        // @ts-ignore
        <CheckButtonPresenter
            {...rest}
            innerRef={
                (typeof innerRef === 'function'
                    ? proxyRefCb(innerRef)
                    : innerRef) as React.RefObject<HTMLElement>
            }
            width={width === 'max' ? 'max' : undefined}
            kind="check"
            view={'default' as 'default'}
            className={cx('CheckButton', className)}
        >
            {buttonContent}
        </CheckButtonPresenter>
    );
};

CheckButton.defaultProps = {
    size: 'm-inset',
    width: 'auto',
    theme: 'secondary',
};

export default CheckButton;
