import React, {
    PureComponent,
    FunctionComponent,
    ReactNode,
    useState,
    useRef,
} from 'react';
import _noop from 'lodash/noop';

import {YANDEX_ROOT_PATH} from 'constants/common';

import {IWithClassName} from 'types/withClassName';
import EPopupDirection from 'components/Popup/types/EPopupDirection';
import {IInputChangeEvent} from 'components/Input/types/InputProps';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {insertJSXIntoKey} from 'utilities/tanker/insertJSXIntoKey';
import {isAuthUser} from 'utilities/userInfo/isAuthUser';

import * as i18nBlock from 'i18nNew/components-CheckOrderAuthorization';

import Button from 'components/Button/Button';
import Input from 'components/Input/Input';
import Spinner from 'components/Spinner/Spinner';
import NotFoundContent from 'components/NotFound/NotFoundContent/NotFoundContent';
import MessageBoxPopup from 'components/MessageBoxPopup/MessageBoxPopup';

import DeviceTypeContext from 'contexts/DeviceTypeContext';

import {ICheckOrderAuthorizationContainer} from './CheckOrderAuthorizationContainer';

import cx from './CheckOrderAuthorization.scss';

export interface ICheckOrderAuthorizationProps
    extends ICheckOrderAuthorizationContainer,
        IWithClassName {
    orderId: string;
    onSuccessOrderAuthorization: Function;
    onFailedOrderAuthorization?: Function;
}

interface ICheckOrderAuthorizationState {
    canRedirectToAuthorization: boolean;
}

interface IOrderAuthorizationFormProps {
    onSubmitForm: Function;
    isSubmittedError: boolean;
}

type toggleErrorTooltipViewType = [
    boolean,
    {
        showErrorTooltip: Function;
        hideErrorTooltip: Function;
    },
];

const useToggleErrorTooltipView = (
    initialState: boolean,
): toggleErrorTooltipViewType => {
    const [canRenderTooltip, setTooltipView] = useState(initialState);
    const showErrorTooltip = (): void => setTooltipView(true);
    const hideErrorTooltip = (): void => setTooltipView(false);

    return [
        canRenderTooltip,
        {
            showErrorTooltip,
            hideErrorTooltip,
        },
    ];
};

const TOOLTIP_DIRECTIONS = [EPopupDirection.TOP, EPopupDirection.BOTTOM];

const OrderAuthorizationForm: FunctionComponent<IOrderAuthorizationFormProps> =
    ({onSubmitForm, isSubmittedError}) => {
        const {isDesktop, isMobile} = useDeviceType();
        const inputRef = useRef<HTMLElement | null>(null);
        const [inputValue, onChangeInputValue] = useState('');
        const [canRenderSubmitError, onToggleSubmitErrorVisible] =
            useState(false);
        const [submittedForm, onTrySubmitForm] = useState(false);
        const [
            canRenderTooltip,
            {showErrorTooltip, hideErrorTooltip},
        ]: toggleErrorTooltipViewType = useToggleErrorTooltipView(false);
        const isEmptyInput = inputValue === '';
        const canRenderErrorState =
            isEmptyInput && canRenderTooltip && submittedForm;

        const handleChangeInput = (
            e: IInputChangeEvent,
            value: string,
        ): void => {
            onChangeInputValue(value);
            onToggleSubmitErrorVisible(false);
        };

        const handleSubmitForm = (
            e: React.FormEvent<HTMLFormElement>,
        ): void => {
            onTrySubmitForm(true);

            if (isEmptyInput) {
                if (inputRef.current) {
                    inputRef.current.focus();
                }
            } else {
                onToggleSubmitErrorVisible(true);
                onSubmitForm({inputValue});
            }

            e.preventDefault();
        };

        return (
            <form onSubmit={handleSubmitForm} noValidate>
                <div className={cx('inputLabel')}>
                    {i18nBlock.emailOrPhoneLabel()}
                </div>
                <Input
                    size={isMobile ? 'l' : 'm'}
                    type="text"
                    state={canRenderErrorState ? 'error' : undefined}
                    value={inputValue}
                    onFocus={() => showErrorTooltip()}
                    onBlur={() => hideErrorTooltip()}
                    onChange={handleChangeInput}
                    inputRef={(currentInputRef): void => {
                        inputRef.current = currentInputRef;
                    }}
                />
                {isMobile && canRenderErrorState && (
                    <div className={cx('errorTooltipLabel')}>
                        {i18nBlock.validateDotEmptyInput()}
                    </div>
                )}
                {isDesktop && inputRef.current && canRenderErrorState && (
                    <MessageBoxPopup
                        isVisible
                        anchorRef={inputRef}
                        direction={TOOLTIP_DIRECTIONS}
                    >
                        <div className={cx('errorTooltip')}>
                            {i18nBlock.validateDotEmptyInput()}
                        </div>
                    </MessageBoxPopup>
                )}
                <Button
                    className={cx('button')}
                    type="submit"
                    size={isMobile ? 'l' : 'm-inset'}
                    width={isMobile ? 'max' : 'auto'}
                    theme="primary"
                >
                    {i18nBlock.submitButton()}
                </Button>
                {canRenderSubmitError && isSubmittedError && (
                    <div className={cx('errorLabel')}>
                        {i18nBlock.createAuthorizationErrorDotInvalidEmailOrPhone()}
                    </div>
                )}
            </form>
        );
    };

class CheckOrderAuthorization extends PureComponent<
    ICheckOrderAuthorizationProps,
    ICheckOrderAuthorizationState
> {
    static readonly defaultProps: Partial<ICheckOrderAuthorizationProps> = {
        className: '',
        onSuccessOrderAuthorization: _noop,
    };

    readonly state: ICheckOrderAuthorizationState = {
        canRedirectToAuthorization: false,
    };

    componentDidMount(): void {
        const {checkOrderAuthorization, orderId} = this.props;

        checkOrderAuthorization(orderId);
        this.allowRedirectToAuthorization();
    }

    componentDidUpdate(
        prevProps: Readonly<ICheckOrderAuthorizationProps>,
    ): void {
        this.checkSuccessOrderAuthorization(prevProps);
    }

    private allowRedirectToAuthorization = (): void => {
        this.setState({canRedirectToAuthorization: true});
    };

    /* Helpers */

    private checkSuccessOrderAuthorization(
        prevProps: Readonly<ICheckOrderAuthorizationProps>,
    ): void {
        const {
            orderAuthorization: {
                checkAuthorization: previousCheckAuthorization,
                createAuthorization: previousCreateAuthorization,
            },
        } = prevProps;

        const {
            orderAuthorization: {checkAuthorization, createAuthorization},
            onSuccessOrderAuthorization,
            onFailedOrderAuthorization,
        } = this.props;

        const isSuccessCheckAuthorization =
            previousCheckAuthorization.isSuccess !==
            checkAuthorization.isSuccess;
        const isSuccessCreateAuthorization =
            previousCreateAuthorization.isSuccess !==
            createAuthorization.isSuccess;

        if (isSuccessCheckAuthorization || isSuccessCreateAuthorization) {
            onSuccessOrderAuthorization();
        } else if (
            createAuthorization.isError &&
            !createAuthorization.isLoading &&
            onFailedOrderAuthorization
        ) {
            onFailedOrderAuthorization();
        }
    }

    private getAuthorizationLink = (): string => {
        const {canRedirectToAuthorization} = this.state;
        const {
            userInfo: {authorizePath},
        } = this.props;

        return `${authorizePath}${
            canRedirectToAuthorization
                ? document.location.href
                : YANDEX_ROOT_PATH
        }`;
    };

    /* Handlers */

    private handleSubmitForm = ({inputValue}: {inputValue: string}): void => {
        const {
            createOrderAuthorization,
            orderAuthorization: {
                checkAuthorization: {orderIds},
            },
        } = this.props;

        if (!orderIds) {
            return;
        }

        createOrderAuthorization({
            id: orderIds.id,
            secret: inputValue,
        });
    };

    /* Render */

    private renderLoader(): ReactNode {
        const {
            orderAuthorization: {
                checkAuthorization: {
                    isLoading: checkLoading,
                    isSuccess: checkSuccess,
                },
                createAuthorization: {
                    isLoading: createLoading,
                    isSuccess: createSuccess,
                },
            },
        } = this.props;

        const awaitStartOrderPollingOrCheckRequests =
            checkLoading || createLoading || checkSuccess || createSuccess;

        return (
            awaitStartOrderPollingOrCheckRequests && (
                <div className={cx('loaderContainer')}>
                    <Spinner />
                </div>
            )
        );
    }

    private renderAuthorizationDescriptionText(): ReactNode {
        const authLink = this.getAuthorizationLink();

        const authorizationLinkNode = (
            <a
                key={authLink}
                className={cx('authorizationLink')}
                href={authLink}
            >
                {i18nBlock.authorizationLink()}
            </a>
        );

        return insertJSXIntoKey(i18nBlock.authorizationDescription)({
            authorizationLinkNode,
        });
    }

    private renderFormContent(): ReactNode {
        const {
            orderAuthorization: {
                checkAuthorization: {orderIds, isError},
                createAuthorization: {isError: isSubmittedError},
            },
            userInfo,
        } = this.props;

        if (!orderIds) {
            return null;
        }

        return (
            isError && (
                <>
                    <h4 className={cx('orderTitle')}>
                        {i18nBlock.orderIdTitle({
                            orderId: orderIds.yandexOrderId,
                        })}
                    </h4>
                    <div className={cx('authorizationDescription')}>
                        {isAuthUser(userInfo)
                            ? i18nBlock.description()
                            : this.renderAuthorizationDescriptionText()}
                    </div>
                    <OrderAuthorizationForm
                        onSubmitForm={this.handleSubmitForm}
                        isSubmittedError={isSubmittedError}
                    />
                </>
            )
        );
    }

    render(): ReactNode {
        const {
            className,
            orderAuthorization: {
                checkAuthorization: {orderIds, isError},
            },
        } = this.props;

        if (Boolean(orderIds?.yandexOrderId) === false && isError) {
            return <NotFoundContent />;
        }

        return (
            <DeviceTypeContext.Consumer>
                {({isMobile}): ReactNode => (
                    <div
                        className={cx(className, 'orderAuthorization', {
                            orderAuthorization_mobile: isMobile,
                        })}
                    >
                        {this.renderLoader()}
                        {this.renderFormContent()}
                    </div>
                )}
            </DeviceTypeContext.Consumer>
        );
    }
}

export default CheckOrderAuthorization;
