import React, {MouseEventHandler, PureComponent, ReactNode} from 'react';

import {IWithClassName} from 'types/withClassName';
import {IWithDeviceType} from 'types/withDeviceType';
import {ECalendarType} from 'components/Calendar/types';

import {
    prepareQaAttributes,
    IWithQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {deviceModMobile} from 'utilities/stylesUtils';
import getPreparedDateForTrigger from 'components/Calendar/utilities/getPreparedDateForTrigger';

import {TDatePickerSize} from 'components/DatePicker/DatePicker';
import SearchControl from 'components/SearchControl/SearchControl';
import CalendarIcon from 'icons/16/Calendar';
import DatePickerIcon from 'icons/16/DatePicker';
import CloseCircleIcon from 'icons/16/CloseCircle';

import cx from './DatePickerTrigger.scss';

export interface IDatePickerTriggerProps
    extends IWithClassName,
        IWithDeviceType,
        IWithQaAttributes {
    focusClassName: string;
    calendarType: ECalendarType;
    placeholder: string;
    visiblePopup: boolean;
    date: Nullable<Date>;
    triggerViewType: 'union' | 'tile';
    size: TDatePickerSize;
    isShortFormatDate: boolean;
    isMobile: boolean;
    canRenderResetIcon: boolean;
    tabIndex?: number;
    hasError: boolean;
    triggerValue?: string;
    onClick?: () => void;
    onMouseDown?: () => void;
    onFocus?: () => void;
    onBlur?: () => void;
    onKeyDown?: () => void;
    onClickReset?: (type: ECalendarType) => void;
}

class DatePickerTrigger extends PureComponent<IDatePickerTriggerProps> {
    readonly buttonRef = React.createRef<HTMLButtonElement>();
    readonly labelRef = React.createRef<HTMLLabelElement>();

    static readonly defaultProps: Partial<IDatePickerTriggerProps> = {
        className: '',
        focusClassName: '',
        placeholder: '',
        triggerViewType: 'union',
        size: 'xl',
        calendarType: ECalendarType.StartDate,
        visiblePopup: false,
        isShortFormatDate: true,
        canRenderResetIcon: true,
        date: null,
        isMobile: false,
        hasError: false,
    };

    focus(): void {
        const {current} = this.buttonRef;

        if (current) {
            current.focus();
        }
    }

    blur(): void {
        const {current} = this.buttonRef;

        if (current) {
            current.blur();
        }
    }

    private handleClickReset: MouseEventHandler = e => {
        const {onClickReset, calendarType} = this.props;

        onClickReset?.(calendarType);

        e.preventDefault();
        e.stopPropagation();

        this.focus();
    };

    private renderValue(): ReactNode {
        const {placeholder, size} = this.props;

        const valueNode = this.renderValueNode();

        if (size === 'xl') {
            return (
                <>
                    {this.renderButton()}
                    <SearchControl
                        placeholderClassName={cx('placeholder')}
                        placeholder={placeholder}
                        controlNode={valueNode}
                        isEmpty={!valueNode}
                    />
                </>
            );
        }

        return this.renderButton(valueNode || this.renderPlaceholderNode());
    }

    private renderValueNode(): ReactNode {
        const {date, triggerValue, isShortFormatDate} = this.props;

        const isEmptyValue = !date && !triggerValue;

        return (
            !isEmptyValue && (
                <div
                    className={cx('value')}
                    {...prepareQaAttributes({
                        parent: this.props,
                        current: 'trigger-value',
                    })}
                >
                    {triggerValue ||
                        getPreparedDateForTrigger(date as Date, {
                            isShortFormatDate,
                        })}
                </div>
            )
        );
    }

    private renderPlaceholderNode(): ReactNode {
        const {placeholder} = this.props;

        return <span className={cx('placeholder')}>{placeholder}</span>;
    }

    private renderResetIcon(): ReactNode {
        const {date, canRenderResetIcon, triggerValue} = this.props;

        if (canRenderResetIcon && (date || triggerValue)) {
            return (
                <div
                    className={cx('resetIcon')}
                    onClick={this.handleClickReset}
                >
                    <CloseCircleIcon
                        className={cx('crossIcon')}
                        width={18}
                        height={18}
                    />
                </div>
            );
        }

        return null;
    }

    private renderDatePickerIcon(): ReactNode {
        const {canRenderResetIcon, visiblePopup, date, triggerValue, size} =
            this.props;

        if (canRenderResetIcon && (date || triggerValue)) {
            return null;
        }

        if (size === 'xl') {
            return (
                <div className={cx('datePickerIcon')}>
                    <DatePickerIcon
                        className={cx('datePickerIcon__icon', {
                            active: visiblePopup,
                        })}
                    />
                </div>
            );
        }

        return <CalendarIcon className={cx('datePickerIcon__icon')} />;
    }

    private renderButton(buttonContent?: ReactNode): ReactNode {
        const {onFocus, onBlur, onKeyDown, tabIndex} = this.props;

        return (
            <button
                className={cx('datePickerTriggerButton')}
                type="button"
                tabIndex={tabIndex}
                onFocus={onFocus}
                onBlur={onBlur}
                onKeyDown={onKeyDown}
                ref={this.buttonRef}
            >
                {buttonContent}
            </button>
        );
    }

    render(): ReactNode {
        const {
            className,
            visiblePopup,
            deviceType,
            calendarType,
            triggerViewType,
            focusClassName,
            hasError,
            size,
            onClick,
            onMouseDown,
        } = this.props;

        return (
            <label
                className={cx(
                    'datePickerTrigger',
                    `datePickerTrigger_${size}`,
                    {
                        datePickerTrigger_focus: visiblePopup,
                        datePickerTrigger_error: hasError,
                        [`datePickerTrigger_viewType_${triggerViewType}`]:
                            triggerViewType,
                        [`datePickerTrigger_type_${calendarType}`]:
                            calendarType,
                    },
                    deviceModMobile('datePickerTrigger', deviceType),
                    className,
                    visiblePopup && focusClassName,
                )}
                ref={this.labelRef}
                onClick={onClick}
                onMouseDown={onMouseDown}
                {...prepareQaAttributes({
                    parent: this.props,
                    current: 'trigger',
                })}
            >
                {this.renderValue()}
                {this.renderDatePickerIcon()}
                {this.renderResetIcon()}
            </label>
        );
    }
}

export default DatePickerTrigger;
