import React from 'react';
import {Helmet} from 'react-helmet';
import _isEmpty from 'lodash/isEmpty';

import {createUniqueReactKeyFromObject} from 'components/SocialSharingMeta/utilities/createUniqueReactKeyFromObject';

interface ICommonMetaTagAttributes {
    content: string;
    name: string;
}

interface ILinkTagAttributes {
    rel: string;
    href: string;
}

export interface IAlternateLink {
    href: string;
    hreflang: string;
}

// Взяты наиболее часто используемые значения отсюда:
// https://yandex.ru/support/webmaster/controlling-robot/meta-robots.html
// https://developers.google.com/search/reference/robots_meta_tag?hl=ru#%D0%B4%D0%B8%D1%80%D0%B5%D0%BA%D1%82%D0%B8%D0%B2%D1%8B
type TMetaRobotsValue =
    | 'noindex'
    | 'nofollow'
    | 'none'
    | 'noyaca'
    | 'noarchive';

interface IDocumentMetaProps {
    description?: string;
    keywords?: string;
    robots?: TMetaRobotsValue | TMetaRobotsValue[];
    /** Массив атрибутов для дополнительных метатегов (будут прокинуты в <meta> в качестве атрибутов метатега) */
    extra?: ICommonMetaTagAttributes[];

    // Ради удобства есть еще пропсы для некоторых других популярных тегов из <head>
    /** Текст в <title> */
    title?: string;
    /** Канонический урл: <link rel="canonical" href="https://travel.yandex.ru/hotels/"> */
    canonicalUrl?: string;
    alternates?: IAlternateLink[];
}

/**
 * Компонент для единообразной работы со всеми <meta>-тегами, КРОМЕ СОЦСЕТЕЙ.
 * @see {SocialSharingMeta} для мета-тегов для СОЦСЕТЕЙ (OGP и прочее)
 * @uses {react-helmet}
 * Вдохновлено https://a.yandex-team.ru/arcadia/data-ui/cloud-www/src/components/DocumentMeta/DocumentMeta.tsx
 **/
class DocumentMeta extends React.Component<IDocumentMetaProps> {
    private static prepareRobotsString(
        robots: IDocumentMetaProps['robots'],
    ): string | undefined {
        if (Array.isArray(robots)) {
            return robots.join(', ');
        }

        return robots;
    }

    private static getUniqueKey(obj: Object): string {
        return createUniqueReactKeyFromObject(obj);
    }

    private static renderLink(
        linkAttrs: ILinkTagAttributes | IAlternateLink,
    ): React.ReactNode {
        return (
            <link key={DocumentMeta.getUniqueKey(linkAttrs)} {...linkAttrs} />
        );
    }

    private static renderMeta(
        metaAttrs: ICommonMetaTagAttributes,
    ): React.ReactNode {
        return (
            <meta key={DocumentMeta.getUniqueKey(metaAttrs)} {...metaAttrs} />
        );
    }

    private static renderTitle(
        title: IDocumentMetaProps['title'],
    ): React.ReactNode {
        return <title>{title}</title>;
    }

    render(): React.ReactNode {
        const {
            title,
            description,
            keywords,
            robots,
            canonicalUrl,
            alternates,
            extra,
        } = this.props;

        const meta = [
            Boolean(description) && {name: 'description', content: description},
            Boolean(keywords) && {name: 'keywords', content: keywords},
            Boolean(robots) && {
                name: 'robots',
                content: DocumentMeta.prepareRobotsString(robots),
            },
        ]
            .concat(extra || [])
            .filter((x): x is ICommonMetaTagAttributes => Boolean(x));

        const links: (ILinkTagAttributes | IAlternateLink)[] = [
            ...(canonicalUrl ? [{rel: 'canonical', href: canonicalUrl}] : []),
            ...(alternates
                ? alternates.map(({href, hreflang}) => ({
                      rel: 'alternate',
                      href,
                      hreflang,
                  }))
                : []),
        ];

        return (
            <Helmet>
                {Boolean(title) && DocumentMeta.renderTitle(title)}
                {!_isEmpty(meta) && meta.map(DocumentMeta.renderMeta)}
                {links.map(DocumentMeta.renderLink)}
            </Helmet>
        );
    }
}

export default DocumentMeta;
