import {ReactNode, FunctionComponent} from 'react';
import {Meta} from '@storybook/react/types-6-0';
import routerDecorator from 'storybook/storyDecorators/routerDecorator';

import {TStorybookFunctionComponent} from 'storybook/types';
import EPopupDirection from 'components/Popup/types/EPopupDirection';

import Text from 'components/Text/Text';
import Input from 'components/Input/Input';
import Button from 'components/Button/Button';
import Calendar, {ICalendarProps} from 'components/Calendar/Calendar';
import CheckButton, {
    ICheckButtonProps,
} from 'components/CheckButton/CheckButton';
import CloseIcon from 'icons/16/Close';

import Dropdown, {
    IDropdownSwitcherParams,
    IDropdownPopupParams,
} from './Dropdown';

export default {
    title: 'Example/Dropdown',
    component: Dropdown,
    decorators: [routerDecorator],
} as Meta;

export const Simple: TStorybookFunctionComponent = () => (
    <Dropdown
        renderSwitcher={({
            input: {ref},
            meta: {visible, hidePopup, showPopup},
        }): ReactNode => (
            <CheckButton
                checked={visible}
                onClick={visible ? hidePopup : showPopup}
                innerRef={ref}
            >
                Справочник путешественника
            </CheckButton>
        )}
        renderPopup={(): ReactNode => (
            <Text size="s">
                В справочнике собраны полезные рекомендации, как находить и
                покупать дешёвые билеты на популярные направления. В нём можно
                узнать, что делать после покупки авиабилета, как легко летать с
                пересадками, как зарегистрироваться на рейс, что можно и что
                нельзя брать с собой в багаж и в самолёт, как правильно
                подготовиться к перелёту и что делать в аэропорту.
            </Text>
        )}
        popupProps={{style: {width: '50%', padding: 20}}}
    />
);
Simple.parameters = {
    info: {
        title: 'Композиция при помощи функций рендеринга попапа и переключателя',
    },
};

// --------------------------------------------------
// --------------------------------------------------
// --------------------------------------------------

const InputWrapper: FunctionComponent<
    IDropdownSwitcherParams<{
        placeholder: string;
    }>
> = ({componentProps, input: {ref, onFocus}}) => (
    <Input {...componentProps} inputRef={ref} onFocus={onFocus} />
);

const CalendarWrapper: FunctionComponent<IDropdownPopupParams<ICalendarProps>> =
    ({componentProps}) => <Calendar {...componentProps} />;

export const TemplateWrapper: TStorybookFunctionComponent = () => (
    <Dropdown
        switcherComponent={InputWrapper}
        switcherComponentProps={{
            placeholder: 'Календарь',
        }}
        popupComponent={CalendarWrapper}
        popupComponentProps={{
            minDate: new Date(2020, 4, 1),
            nowDate: new Date(2020, 4, 20),
            maxDate: new Date(2020, 4, 30),
            isRenderMonthsList: true,
            lastScrollSource: null,
            scrollPartPosition: 0,
        }}
        popupProps={{
            directions: [EPopupDirection.BOTTOM_LEFT],
        }}
    />
);
TemplateWrapper.parameters = {
    info: {
        title: 'Композиция подготовленных компонентов попапа и переключателя',
    },
};

// --------------------------------------------------
// --------------------------------------------------
// --------------------------------------------------

const Switcher: FunctionComponent<
    IDropdownSwitcherParams<Partial<ICheckButtonProps>>
> = ({componentProps, input: {ref}, meta: {visible, showPopup}}) => (
    <CheckButton
        checked={visible}
        onClick={showPopup}
        innerRef={ref}
        {...componentProps}
    />
);

const ModalPopup: FunctionComponent<IDropdownPopupParams<ICalendarProps>> = ({
    componentProps,
    meta: {hidePopup},
}) => (
    <div style={{background: 'white', height: '100vh'}}>
        <div style={{maxWidth: 510, margin: 'auto'}}>
            <Button width="max" icon={<CloseIcon />} onClick={hidePopup}>
                Закрыть
            </Button>
            <Calendar {...componentProps} />
        </div>
    </div>
);

export const DefaultPopup: TStorybookFunctionComponent = () => (
    <Dropdown
        switcherComponent={Switcher}
        switcherComponentProps={{children: 'Открыть модальное окно'}}
        isModalView={true}
        popupComponent={ModalPopup}
        popupComponentProps={{
            minDate: new Date(2020, 4, 1),
            nowDate: new Date(2020, 4, 20),
            maxDate: new Date(2020, 4, 30),
            isRenderMonthsList: true,
            lastScrollSource: null,
            scrollPartPosition: 0,
        }}
    />
);
DefaultPopup.parameters = {
    info: {
        title: 'Вариант с модальным попапом',
    },
};
