import React, {useCallback, useRef} from 'react';
import Foco from 'react-foco';

import EPopupDirection from 'components/Popup/types/EPopupDirection';

import {useBoolean} from 'utilities/hooks/useBoolean';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import CheckButton from 'components/CheckButton/CheckButton';
import Popup from 'components/Popup/Popup';

import cx from './DropdownFilter.scss';

interface IDropdownFilterProps extends IWithQaAttributes {
    popupClassName?: string;
    label: React.ReactNode;
    checked?: boolean;
    children: React.ReactNode;
    isNarrow?: boolean;
    onClickButton?: () => void; // Клик по кнопке-открывашке
}

const ARROWS_ICON = (
    <svg className={cx('icon')} viewBox="0 0 26 24" fill="none">
        <path d="M23 5l3 6h-6l3-6zM23 19l3-6h-6l3 6z" fill="currentColor" />
    </svg>
);

const DIRECTIONS = [EPopupDirection.BOTTOM_LEFT, EPopupDirection.BOTTOM_RIGHT];

const DropdownFilter: React.FC<IDropdownFilterProps> = props => {
    const {
        label,
        children,
        checked,
        popupClassName,
        isNarrow = false,
        onClickButton,
    } = props;

    const {
        value: visible,
        setFalse: hidePopup,
        setTrue: showPopup,
    } = useBoolean(false);

    const handlerClickCheckButton = useCallback(() => {
        if (visible) {
            hidePopup();
        } else {
            showPopup();
        }

        if (onClickButton) {
            onClickButton();
        }
    }, [hidePopup, onClickButton, showPopup, visible]);

    const anchorRef = useRef<HTMLElement>(null);
    const containerRef = useRef<HTMLElement>(null);

    return (
        <Foco onClickOutside={hidePopup} onFocusOutside={hidePopup}>
            {focoProps => (
                <span
                    className={cx('trigger-container', {
                        'trigger-container_narrow': isNarrow,
                    })}
                    ref={containerRef}
                    {...focoProps}
                >
                    <CheckButton
                        {...prepareQaAttributes(props)}
                        checked={Boolean(checked)}
                        size="m-inset"
                        innerRef={anchorRef}
                        onClick={handlerClickCheckButton}
                    >
                        {label}
                        {ARROWS_ICON}
                    </CheckButton>

                    <Popup
                        className={cx('popup', popupClassName)}
                        visible={visible}
                        scope={containerRef}
                        anchor={anchorRef}
                        directions={DIRECTIONS}
                        zIndex={10}
                    >
                        {children}
                    </Popup>
                </span>
            )}
        </Foco>
    );
};

export default React.memo(DropdownFilter);
