import React from 'react';
import _chunk from 'lodash/chunk';

import {IWithClassName} from 'types/withClassName';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {deviceMods} from 'utilities/stylesUtils';
import IPrice from 'utilities/currency/PriceInterface';
import {insertJSXIntoKey} from 'utilities/tanker/insertJSXIntoKey';

import * as i18nBlock from 'i18nNew/components-FeaturedDirections';
import * as i18nMarkupBlock from 'i18nNew/buses-indexPage';

import {CardWithDeviceLayout} from 'components/CardWithDeviceLayout/CardWithDeviceLayout';
import Heading from 'components/Heading/Heading';
import Link from 'components/Link/Link';
import Price from 'components/Price/Price';
import Box from 'components/Box/Box';
import Flex from 'components/Flex/Flex';

import cx from './FeaturedDirections.scss';

const POPULAR_DIRECTIONS_COLUMN_COUNT = 5;

interface IFeaturedDirectionsItem {
    text: string;
    url: string;
    price?: IPrice | null;
}

interface IFeaturedDirectionsProps extends IWithClassName {
    title: string;
    items: IFeaturedDirectionsItem[];
    morePopularUrl?: string;
}

const FeaturedDirections: React.FC<IFeaturedDirectionsProps> = ({
    title,
    items,
    className,
    morePopularUrl,
}) => {
    const deviceType = useDeviceType();

    if (!items || !items.length) {
        return null;
    }

    const directionNodes = (): React.ReactNodeArray => {
        return _chunk(items, POPULAR_DIRECTIONS_COLUMN_COUNT).map(
            (column, index) => (
                <Box key={index} className={cx('column')} between={3}>
                    {column.map(({text, url, price}) => (
                        <Flex
                            key={url}
                            between={deviceType.isDesktop ? 6 : 5}
                            justifyContent="space-between"
                            alignItems="baseline"
                            textSize="m"
                            inline
                            nowrap
                        >
                            <Link
                                target="_blank"
                                to={{
                                    pathname: url,
                                    state: {
                                        scrollTop: 0,
                                    },
                                }}
                            >
                                {text}
                            </Link>
                            {price && (
                                <Box
                                    className={cx('price')}
                                    textWeight="bold"
                                    textColor="secondary"
                                >
                                    {insertJSXIntoKey(
                                        i18nMarkupBlock.popularDashPrice,
                                    )({
                                        price: (
                                            <Price {...price} isRound={true} />
                                        ),
                                    })}
                                </Box>
                            )}
                        </Flex>
                    ))}
                </Box>
            ),
        );
    };

    return (
        <CardWithDeviceLayout
            className={cx('root', deviceMods('root', deviceType), className)}
        >
            <Heading level={2}>{title}</Heading>
            <Box
                above={7}
                between={deviceType.isDesktop ? 10 : 3}
                inline={deviceType.isDesktop}
                nowrap
            >
                {directionNodes()}
            </Box>
            {morePopularUrl && (
                <Box above={7}>
                    <Link
                        className={cx('morePopular')}
                        target="_blank"
                        to={{
                            pathname: morePopularUrl,
                        }}
                    >
                        <strong>{i18nBlock.morePopular()}</strong>
                    </Link>
                </Box>
            )}
        </CardWithDeviceLayout>
    );
};

export default React.memo(FeaturedDirections);
