import React from 'react';
import identity from 'lodash/identity';

import {IWithClassName} from 'types/withClassName';
import {IWithDeviceType} from 'types/withDeviceType';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import Checkbox from 'components/Checkbox/Checkbox';
import Flex from 'components/Flex/Flex';

import cx from './FilterCheckList.scss';

export interface IFilterCheckListProps<O, V>
    extends IWithClassName,
        IWithQaAttributes,
        IWithDeviceType {
    /**
     * Выбранные значения
     */
    values: V[];
    /**
     * Значения доступные для выбора, остальные блокируются
     */
    availableValues?: V[];

    /**
     * Все варианты в списке
     */
    options: O[];
    /**
     * Сопоставляет option - значению
     */
    optionToValue?: (option: O) => V;

    /**
     * Функция для рендера текста у option
     */
    renderText?: (option: O) => React.ReactNode;

    /**
     * Серая подпись справа от пункта
     */
    renderComment?: (option: O) => React.ReactNode;

    /**
     * Подпись справа
     * можно например указать цену
     */
    renderAside?: (option: O, index?: number) => React.ReactNode;

    /**
     * Колбэк на изменение чекбоксов
     * @param option - на котором произошло изменение
     * @param checked - новое состояние галочки
     */
    onChange: (option: O, checked: boolean) => void;
}

export default function FilterCheckList<O, V>({
    className,
    options,
    optionToValue = identity,
    values,
    availableValues,
    renderText = identity,
    renderComment,
    renderAside,
    deviceType,
    onChange,
    ...rest
}: IFilterCheckListProps<O, V>): React.ReactElement {
    return (
        <Flex
            className={className}
            flexDirection="column"
            between={deviceType.isMobile ? 6 : 2}
            {...prepareQaAttributes(rest)}
        >
            {options.map((option, index) => {
                const value = optionToValue(option);
                const valueStr = String(value);
                const isChecked = values.includes(value);
                const itemQa = prepareQaAttributes({
                    parent: rest,
                    current: 'option',
                    key: valueStr,
                });

                return (
                    <Checkbox
                        key={valueStr}
                        checked={isChecked}
                        value={valueStr}
                        label={
                            <Flex
                                justifyContent="space-between"
                                {...prepareQaAttributes(itemQa)}
                            >
                                <div
                                    className={cx('text')}
                                    {...prepareQaAttributes({
                                        parent: itemQa,
                                        current: 'text',
                                    })}
                                >
                                    {renderText(option)}
                                </div>
                                {renderComment && (
                                    <div
                                        className={cx('comment')}
                                        {...prepareQaAttributes({
                                            parent: itemQa,
                                            current: 'comment',
                                        })}
                                    >
                                        {renderComment(option)}
                                    </div>
                                )}
                                {renderAside && (
                                    <div
                                        className={cx('aside')}
                                        {...prepareQaAttributes({
                                            parent: itemQa,
                                            current: 'aside',
                                        })}
                                    >
                                        {renderAside(option, index)}
                                    </div>
                                )}
                            </Flex>
                        }
                        disabled={
                            availableValues
                                ? !isChecked && !availableValues.includes(value)
                                : false
                        }
                        onChange={(event): void =>
                            onChange(option, event.target.checked)
                        }
                        {...prepareQaAttributes({
                            parent: itemQa,
                            current: 'checkbox',
                        })}
                    />
                );
            })}
        </Flex>
    );
}
