import React, {FunctionComponent, useCallback} from 'react';

import {IWithClassName} from 'types/withClassName';

import {useBoolean} from 'utilities/hooks/useBoolean';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import * as i18nCommonFiltersBlock from 'i18nNew/common-filters';

import CheckButton from 'components/CheckButton/CheckButton';
import FilterIcon16 from 'icons/16/Filter';
import FilterIcon24 from 'icons/24/Filter';
import CornerCounter from 'components/CornerCounter/CornerCounter';
import ModalWithBackButton from 'components/ModalWithBackButton/ModalWithBackButton';
import ModalContent from 'components/FiltersAndSortsMobile/components/ModalContent/ModalContent';

import cx from './FiltersAndSortsMobile.scss';

export interface IFiltersAndSortsMobileProps
    extends IWithClassName,
        IWithQaAttributes {
    /**
     * Количество сегментов, видимых после фильтрации.
     */
    itemsCount: number;

    /**
     * Количество активных фильтров.
     */
    activeFiltersCount: number;

    /**
     * Нужно ли отображать кнопку узкой и без тексты "Фильтры".
     */
    isCompact: boolean;

    /**
     * Скелетон, отображаемый во время загрузки
     */
    skeleton?: React.ReactElement | null;

    /**
     * Контент модала.
     */
    children: React.ReactNode;

    /**
     * Активна ли кнопка сброса
     */
    canReset?: boolean;

    /**
     * Обработчик применения фильтров.
     */
    onApply?(): void;

    /**
     * Обработчик сброса фильтров.
     */
    onReset?(): void;
}

const FiltersAndSortsMobile: FunctionComponent<IFiltersAndSortsMobileProps> =
    props => {
        const {
            className,
            activeFiltersCount,
            itemsCount,
            isCompact,
            skeleton,
            children,
            canReset = true,
            onApply,
            onReset,
        } = props;

        const {
            value: isOpened,
            setTrue: open,
            setFalse: close,
        } = useBoolean(false);

        const handleApply = useCallback(() => {
            close();

            onApply?.();
        }, [close, onApply]);

        const handleReset = useCallback(() => {
            onReset?.();
        }, [onReset]);

        return (
            skeleton || (
                <>
                    <CornerCounter
                        className={cx(
                            'cornerCounter',
                            {
                                cornerCounter_compact: isCompact,
                            },
                            className,
                        )}
                        count={activeFiltersCount}
                    >
                        <CheckButton
                            icon={
                                isCompact ? (
                                    <FilterIcon24 width={24} height={24} />
                                ) : (
                                    <FilterIcon16 width={16} height={16} />
                                )
                            }
                            checked={activeFiltersCount > 0}
                            onClick={open}
                            size={isCompact ? 'xl' : 'l'}
                            width="max"
                            {...prepareQaAttributes({
                                parent: props,
                                current: 'toggleButton',
                            })}
                        >
                            {!isCompact && i18nCommonFiltersBlock.title()}
                        </CheckButton>
                    </CornerCounter>

                    <ModalWithBackButton
                        containerClassName={cx('modalContainer')}
                        isVisible={isOpened}
                        onClose={close}
                        canRenderBackButton
                        fullScreen
                        preventBodyScroll={isOpened}
                        verticalAlign="stretch"
                    >
                        <ModalContent
                            itemsCount={itemsCount}
                            onApply={handleApply}
                            canReset={canReset}
                            onReset={handleReset}
                            {...prepareQaAttributes(props)}
                        >
                            {children}
                        </ModalContent>
                    </ModalWithBackButton>
                </>
            )
        );
    };

export default React.memo(FiltersAndSortsMobile);
