import {Decorator, Unsubscribe, FormApi} from 'final-form';
import _isEmpty from 'lodash/isEmpty';

import {TFocusFirstInvalidField} from '../mutators/focusFirstInvalidField';

/**
 * Декоратор, который добавляет ошибки валидации при существующих Blur ошибках.
 * Нужен из-за того, что FinalForm не вызывает Submit handler -> submit валидацию,
 * до тех пор пока есть хоть одна ошибка Blur валидации
 *
 * Для работы требует мутаторы из src/components/Form/mutators/setFormErrors.ts
 */
const createDecorator =
    <T>(
        formName: string,
        onSubmit?: (values: T, errors: object) => void,
    ): Decorator<T> =>
    (form: FormApi<T>): Unsubscribe => {
        const originalSubmit = form.submit;

        // Monkey patch submit function
        form.submit = (): Promise<T | undefined> | undefined => {
            const result = originalSubmit.call(form);
            const formState = form.getState();
            const {values, errors, submitErrors} = formState;
            const fieldNames = form.getRegisteredFields();

            fieldNames.forEach(fieldName => {
                const {data, value} =
                    form.getFieldState(fieldName as any) || {};
                const submitAmount = (data?.submitAmount || 0) + 1;

                form.mutators.setFieldData?.(fieldName, {
                    lastSubmittedValue: value,
                    submitAmount: submitAmount,
                });
            });

            if (form.mutators.focusFirstInvalidField && !_isEmpty(errors)) {
                (
                    form.mutators
                        .focusFirstInvalidField as TFocusFirstInvalidField
                )(formName, errors);
            } else if (
                form.mutators.focusFirstInvalidField &&
                !_isEmpty(submitErrors)
            ) {
                (
                    form.mutators
                        .focusFirstInvalidField as TFocusFirstInvalidField
                )(formName, submitErrors);
            }

            if (onSubmit) {
                onSubmit(values, errors);
            }

            return result;
        };

        return (): void => {
            form.submit = originalSubmit;
        };
    };

export default createDecorator;
