import React, {Component} from 'react';
import _throttle from 'lodash/throttle';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import cx from './GoToForm.scss';

const SCROLL_THROTTLE_TIME = 100;

export interface IGoToFormProps extends IWithQaAttributes {
    getBorderTopToHidden: () => number | undefined;
    active: boolean;
    children: React.ReactNode;
}

interface IGoToFormState {
    aboveBorderTopToHidden: boolean;
}

export default class GoToForm extends Component<
    IGoToFormProps,
    IGoToFormState
> {
    readonly state: IGoToFormState = {
        aboveBorderTopToHidden: true,
    };

    private readonly rootRef = React.createRef<HTMLDivElement>();

    private checkVisibilityPriceInfoWithThrottle = _throttle(() => {
        const {getBorderTopToHidden} = this.props;
        const borderTopToHidden = getBorderTopToHidden();

        if (!borderTopToHidden || this.height === undefined) {
            return;
        }

        const aboveBorderTopToHidden =
            window.pageYOffset +
                document.documentElement.clientHeight -
                this.height <
            borderTopToHidden;

        if (this.state.aboveBorderTopToHidden !== aboveBorderTopToHidden) {
            this.setState({aboveBorderTopToHidden});
        }
    }, SCROLL_THROTTLE_TIME);

    componentDidMount(): void {
        if (!this.rootRef.current) {
            return;
        }

        this.height = this.rootRef.current.getBoundingClientRect().height;
        window.addEventListener(
            'scroll',
            this.checkVisibilityPriceInfoWithThrottle,
        );

        this.checkVisibilityPriceInfoWithThrottle();
    }

    componentWillUnmount(): void {
        window.removeEventListener(
            'scroll',
            this.checkVisibilityPriceInfoWithThrottle,
        );
    }

    private height: number | undefined;

    render(): React.ReactNode {
        const {active, children} = this.props;
        const {aboveBorderTopToHidden} = this.state;

        if (!active || !aboveBorderTopToHidden) {
            return null;
        }

        return (
            <div
                className={cx('root')}
                ref={this.rootRef}
                {...prepareQaAttributes(this.props)}
            >
                {children}
            </div>
        );
    }
}
